/**
 * ui_messages_frequency.js
 *
 * Adds message frequency analysis functionality to the messages view.
 * - Click any message to analyze its frequency pattern
 * - Shows histogram of time intervals between matching messages
 * - Uses Chart.js for visualization with mean/median indicators
 * - Matches on: src_add, dest_add, src_ep, dest_ep, and payload prefix (2 bytes)
 * 
 * See README.md for extension patterns.
 */

// Debug flag for frequency analysis
window.MESSAGE_FREQUENCY_DEBUG = false;

// Global frequency analyzer object
const messageFrequencyAnalyzer = {
    // Configuration
    config: {
        payloadPrefixBytes: 2,  // Match first 2 bytes of payload
        // Endpoint pairs where we DO payload matching (all others skip payload)
        enablePayloadForEndpoints: [
            { src_ep: 160, dest_ep: 108 }   // Evertag specific - match payload type
        ],
        chartColors: {
            barFill: 'rgba(76, 175, 80, 0.6)',
            barBorder: 'rgb(76, 175, 80)',
            meanLine: 'rgb(255, 152, 0)',
            medianLine: 'rgb(33, 150, 243)'
        }
    },

    // State
    chartInstance: null,
    currentSignature: null,
    currentMessageData: null,  // Store message data for later analysis
    currentMatchingMessages: null,  // Store matching messages
    currentHistogramData: null,  // Store histogram data with message pairs for click handling

    /**
     * Initialize the frequency analyzer
     */
    initialize: function() {
        console.log('[Frequency] Initializing message frequency analyzer...');
        
        // Add message click handlers
        this.addMessageClickHandlers();
        
        // Create modal structure
        this.createModal();
        
        // Initialize buffer counter
        this.initializeBufferCounter();
        
        console.log('[Frequency] Message frequency analyzer initialized');
    },

    /**
     * Initialize and start updating the buffer message counter
     */
    initializeBufferCounter: function() {
        // Update counter immediately
        this.updateBufferCounter();
        
        // Update every 2 seconds
        setInterval(() => {
            this.updateBufferCounter();
        }, 2000);
        
        console.log('[Frequency] Buffer counter initialized');
    },

    /**
     * Update the buffer message counter display
     */
    updateBufferCounter: function() {
        const counterElement = document.getElementById('buffer_message_count');
        const selectedElement = document.getElementById('buffer_selected_count');
        if (!counterElement) return;
        
        try {
            if (typeof mainApp !== 'undefined' && typeof mainApp.getAllMessages === 'function') {
                const messages = mainApp.getAllMessages();
                const count = messages.length;
                counterElement.textContent = count.toLocaleString(); // Format with commas
                
                // Change color based on count
                if (count === 0) {
                    counterElement.style.color = '#999';
                } else if (count < 100) {
                    counterElement.style.color = '#FF9800'; // Orange - low count
                } else if (count < 1000) {
                    counterElement.style.color = '#4CAF50'; // Green - good count
                } else {
                    counterElement.style.color = '#2196F3'; // Blue - large count
                }
                
                // Update selected message count if there's a current signature
                if (selectedElement && this.currentSignature) {
                    const matchingMessages = this.filterMatchingMessages(messages, this.currentSignature);
                    const matchCount = matchingMessages.length;
                    
                    // Store for later use
                    this.currentMatchingMessages = matchingMessages;
                    
                    const matchSpan = selectedElement.querySelector('#buffer_selected_number');
                    const textSpan = selectedElement.querySelector('#buffer_selected_text');
                    const analyzerLink = document.getElementById('frequency_analyzer_link');
                    
                    if (matchSpan && textSpan) {
                        matchSpan.textContent = matchCount.toLocaleString();
                        
                        // Update text based on count
                        if (matchCount === 1) {
                            textSpan.textContent = 'similar message';
                        } else {
                            textSpan.textContent = 'similar messages';
                        }
                        
                        // Color code based on match count
                        if (matchCount === 0) {
                            matchSpan.style.color = '#f44336'; // Red - no matches (shouldn't happen)
                        } else if (matchCount === 1) {
                            matchSpan.style.color = '#FF9800'; // Orange - only 1 (unique message)
                        } else if (matchCount < 10) {
                            matchSpan.style.color = '#FFC107'; // Amber - few matches
                        } else {
                            matchSpan.style.color = '#4CAF50'; // Green - good matches
                        }
                    }
                    
                    // Show/hide analyzer link based on match count
                    if (analyzerLink) {
                        if (matchCount >= 2) {
                            analyzerLink.style.display = 'inline';
                        } else {
                            analyzerLink.style.display = 'none';
                        }
                    }
                    
                    selectedElement.style.display = 'inline';
                } else if (selectedElement) {
                    selectedElement.style.display = 'none';
                    const analyzerLink = document.getElementById('frequency_analyzer_link');
                    if (analyzerLink) {
                        analyzerLink.style.display = 'none';
                    }
                }
            } else {
                counterElement.textContent = 'N/A';
                counterElement.style.color = '#999';
                if (selectedElement) {
                    selectedElement.style.display = 'none';
                }
            }
        } catch (error) {
            console.error('[Frequency] Error updating buffer counter:', error);
        }
    },

    /**
     * Add click handlers to messages in the list
     */
    addMessageClickHandlers: function() {
        // Try multiple possible selectors for the message list
        const possibleSelectors = [
            '#message_list .list',
            '#message_list ul',
            '#message_list',
            '.messages-window .list',
            '.messages-window ul'
        ];
        
        let messageList = null;
        for (const selector of possibleSelectors) {
            messageList = document.querySelector(selector);
            if (messageList) {
                console.log(`[Frequency] Found message list using selector: ${selector}`);
                break;
            }
        }
        
        if (!messageList) {
            console.warn('[Frequency] Message list not found, retrying in 500ms...');
            console.log('[Frequency] Attempted selectors:', possibleSelectors);
            setTimeout(() => this.addMessageClickHandlers(), 500);
            return;
        }

        // Use event delegation for performance and dynamic content
        const clickHandler = (event) => {
            console.log('[Frequency] Click detected on message list');
            console.log('[Frequency] Click target:', event.target);
            console.log('[Frequency] Click target tagName:', event.target.tagName);
            console.log('[Frequency] Click target classList:', event.target.classList);
            
            // Try to find the message item using various strategies
            let messageItem = null;
            
            // Strategy 1: Find closest li element
            messageItem = event.target.closest('li');
            
            // Strategy 2: If target is li itself
            if (!messageItem && event.target.tagName === 'LI') {
                messageItem = event.target;
            }
            
            // Strategy 3: Look for any element that contains message fields
            if (!messageItem) {
                // Check if clicked element or its parent has message-related classes/data
                let el = event.target;
                for (let i = 0; i < 5; i++) {  // Search up to 5 levels
                    if (!el) break;
                    
                    // Check if this element has message-related content
                    if (el.querySelector('.src_add, .dest_add, .src_ep, .dest_ep')) {
                        messageItem = el;
                        console.log('[Frequency] Found message item via child elements at level', i);
                        break;
                    }
                    
                    // Check if element has typical message classes
                    if (el.classList && (el.classList.contains('message') || 
                        el.querySelector('[class*="terminal"]'))) {
                        messageItem = el;
                        console.log('[Frequency] Found message item via class at level', i);
                        break;
                    }
                    
                    el = el.parentElement;
                }
            }
            
            // Strategy 4: Try to find by looking for spans with terminal_field class
            if (!messageItem) {
                let el = event.target;
                while (el && el !== messageList) {
                    if (el.querySelector('.terminal_field')) {
                        messageItem = el;
                        console.log('[Frequency] Found message item via terminal_field search');
                        break;
                    }
                    el = el.parentElement;
                }
            }
            
            if (!messageItem) {
                console.log('[Frequency] No message item found from click');
                console.log('[Frequency] Event target parent chain:', event.target.parentElement);
                return;
            }

            console.log('[Frequency] Message item clicked:', messageItem);

            // Highlight selected message
            // Try to find all message items (could be li, p, div, etc.)
            const allMessages = messageList.querySelectorAll('li, p, div[class*="terminal"], .message');
            allMessages.forEach(msg => msg.classList.remove('frequency-selected'));
            messageItem.classList.add('frequency-selected');

            // Find the message in buffer by matching the DOM element
            const messageData = this.findMessageInBufferByDOM(messageItem);
            if (messageData) {
                this.selectMessage(messageData);
            } else {
                console.error('[Frequency] Could not find message in buffer for clicked item');
            }
        };
        
        messageList.addEventListener('click', clickHandler);

        // Add CSS for clickable messages
        this.addMessageStyles();
        
        const messageCount = messageList.querySelectorAll('li').length;
        console.log(`[Frequency] Click handlers added to message list (${messageCount} messages found)`);
        
        // Log success message to help user verify
        console.log('%c[Frequency Analyzer] Ready! Click any message to analyze its frequency.', 
                   'color: #4CAF50; font-weight: bold; font-size: 14px;');
    },

    /**
     * Add CSS styles for clickable messages
     */
    addMessageStyles: function() {
        // Remove existing style if present
        const existingStyle = document.getElementById('message_frequency_styles');
        if (existingStyle) {
            existingStyle.remove();
        }
        
        const style = document.createElement('style');
        style.id = 'message_frequency_styles';
        style.textContent = `
            /* Make all potential message elements clickable */
            #message_list .list > *,
            #message_list ul > *,
            #message_list > * {
                cursor: pointer !important;
                transition: background-color 0.2s;
            }
            #message_list .list > *:hover,
            #message_list ul > *:hover,
            #message_list > *:hover {
                background-color: rgba(76, 175, 80, 0.1) !important;
            }
            .frequency-selected {
                background-color: rgba(76, 175, 80, 0.2) !important;
                border-left: 4px solid #4CAF50 !important;
                padding-left: 8px !important;
            }
        `;
        document.head.appendChild(style);
        console.log('[Frequency] Styles added for clickable messages');
    },

    /**
     * Find message in buffer by matching DOM element data
     * This is more reliable than extracting from DOM classes
     * @param {HTMLElement} messageItem - The message list item
     * @returns {Object|null} - Message data object from buffer
     */
    findMessageInBufferByDOM: function(messageItem) {
        try {
            // Get all messages from buffer
            const allMessages = this.getAllMessagesFromBuffer();
            if (!allMessages || allMessages.length === 0) {
                console.error('[Frequency] No messages in buffer');
                return null;
            }

            // Extract timestamp from DOM (most reliable unique identifier)
            const timestampEl = messageItem.querySelector('.rx_ts');
            if (timestampEl) {
                const timestampText = timestampEl.textContent.trim();
                console.log('[Frequency] Looking for message with timestamp:', timestampText);
                
                // Try to match by timestamp
                for (let msg of allMessages) {
                    const msgTimestamp = this.formatTimestampForDisplay(msg.rx_ts);
                    if (msgTimestamp === timestampText) {
                        console.log('[Frequency] Found matching message by timestamp:', msg);
                        return msg;
                    }
                }
            }

            // Fallback: Try to match by position
            // Get the index of this message in the visible list
            const messageList = document.querySelector('#message_list ul, #message_list .list');
            if (messageList) {
                const allVisibleMessages = Array.from(messageList.children);
                const messageIndex = allVisibleMessages.indexOf(messageItem);
                
                if (messageIndex >= 0 && messageIndex < allMessages.length) {
                    console.log(`[Frequency] Matched by index ${messageIndex}:`, allMessages[messageIndex]);
                    return allMessages[messageIndex];
                }
            }

            console.error('[Frequency] Could not match DOM element to buffer message');
            return null;
        } catch (error) {
            console.error('[Frequency] Error finding message in buffer:', error);
            return null;
        }
    },

    /**
     * Format timestamp for display matching
     * @param {string|number} timestamp - Timestamp value
     * @returns {string} - Formatted timestamp
     */
    formatTimestampForDisplay: function(timestamp) {
        // This should match the format used in the UI
        // Try to parse and format consistently
        if (!timestamp) return '';
        
        try {
            // If it's already a formatted string, return it
            if (typeof timestamp === 'string' && timestamp.includes('/')) {
                return timestamp;
            }
            
            // If it's a number, convert it to date
            const date = new Date(parseFloat(timestamp));
            if (!isNaN(date.getTime())) {
                // Match the format shown in UI: "2025/9/30 23:21:13.553"
                const year = date.getFullYear();
                const month = date.getMonth() + 1;
                const day = date.getDate();
                const hours = date.getHours().toString().padStart(2, '0');
                const minutes = date.getMinutes().toString().padStart(2, '0');
                const seconds = date.getSeconds().toString().padStart(2, '0');
                const ms = date.getMilliseconds();
                
                return `${year}/${month}/${day} ${hours}:${minutes}:${seconds}.${ms}`;
            }
            
            return String(timestamp);
        } catch (error) {
            return String(timestamp);
        }
    },

    /**
     * Extract message data from DOM element (DEPRECATED - kept for reference)
     * @param {HTMLElement} messageItem - The message list item
     * @returns {Object|null} - Message data object
     */
    extractMessageDataFromDOM: function(messageItem) {
        try {
            const data = {};

            // Extract source address
            const srcAddEl = messageItem.querySelector('.src_add');
            if (srcAddEl) {
                const srcText = srcAddEl.textContent.trim();
                // Handle both decimal and hex formats
                data.src_add = srcText.startsWith('0x') ? parseInt(srcText, 16) : parseInt(srcText);
            }

            // Extract destination address
            const destAddEl = messageItem.querySelector('.dest_add');
            if (destAddEl) {
                const destText = destAddEl.textContent.trim();
                data.dest_add = destText.startsWith('0x') ? parseInt(destText, 16) : parseInt(destText);
            }

            // Extract source endpoint
            const srcEpEl = messageItem.querySelector('.src_ep');
            if (srcEpEl) {
                data.src_ep = parseInt(srcEpEl.textContent.trim());
            }

            // Extract destination endpoint
            const destEpEl = messageItem.querySelector('.dest_ep');
            if (destEpEl) {
                data.dest_ep = parseInt(destEpEl.textContent.trim());
            }

            // Extract payload - look for hex data
            const payloadText = this.findPayloadInMessage(messageItem);
            if (payloadText) {
                data.payload = payloadText;
                try {
                    data.payloadPrefix = this.extractPayloadPrefix(payloadText);
                } catch (error) {
                    console.warn('[Frequency] Error extracting payload prefix from DOM:', error);
                    data.payloadPrefix = '';
                }
            }

            // Extract timestamp if available
            const timestampEl = messageItem.querySelector('.rx_ts');
            if (timestampEl) {
                data.rx_ts = timestampEl.textContent.trim();
            }

            if (window.MESSAGE_FREQUENCY_DEBUG) {
                console.log('[Frequency] Extracted message data:', data);
            }

            return data;
        } catch (error) {
            console.error('[Frequency] Error extracting message data:', error);
            return null;
        }
    },

    /**
     * Find payload data in message item
     * @param {HTMLElement} messageItem - The message element
     * @returns {string|null} - Payload hex string
     */
    findPayloadInMessage: function(messageItem) {
        // Look for payload in various possible locations
        const payloadEl = messageItem.querySelector('.payload, .terminal_field.payload, [class*="payload"]');
        if (payloadEl) {
            const text = payloadEl.textContent.trim();
            // Check if it looks like hex data
            if (this.isHexPayload(text)) {
                return text;
            }
        }

        // Fallback: search all text content for hex pattern
        const allText = messageItem.textContent;
        const hexMatch = allText.match(/([0-9A-Fa-f]{2}\s*)+/);
        if (hexMatch) {
            return hexMatch[0].trim();
        }

        return null;
    },

    /**
     * Check if text looks like hex payload
     * @param {string} text - Text to check
     * @returns {boolean}
     */
    isHexPayload: function(text) {
        if (!text || text.length < 2) return false;
        // Check if it's hex pairs separated by spaces
        const hexPairs = text.trim().split(/\s+/);
        return hexPairs.length > 0 && hexPairs.every(pair => /^[0-9A-Fa-f]{2}$/.test(pair));
    },

    /**
     * Extract payload prefix (first N bytes)
     * @param {string|Array} payloadHex - Full payload hex string or array
     * @returns {string} - Payload prefix
     */
    extractPayloadPrefix: function(payloadHex) {
        if (!payloadHex) return '';
        
        // Handle if payload is an array
        if (Array.isArray(payloadHex)) {
            const prefix = payloadHex.slice(0, this.config.payloadPrefixBytes)
                .map(byte => {
                    if (typeof byte === 'number') {
                        return byte.toString(16).padStart(2, '0').toUpperCase();
                    }
                    return String(byte).toUpperCase();
                })
                .join(' ');
            return prefix;
        }
        
        // Handle if payload is Uint8Array or similar
        if (payloadHex.constructor && payloadHex.constructor.name === 'Uint8Array') {
            const prefix = Array.from(payloadHex)
                .slice(0, this.config.payloadPrefixBytes)
                .map(byte => byte.toString(16).padStart(2, '0').toUpperCase())
                .join(' ');
            return prefix;
        }
        
        // Handle as string
        const payloadStr = String(payloadHex).trim();
        const hexPairs = payloadStr.split(/\s+/);
        const prefix = hexPairs.slice(0, this.config.payloadPrefixBytes).join(' ');
        return prefix.toUpperCase();
    },

    /**
     * Check if payload matching should be skipped for these endpoints
     * Only endpoints in the enablePayloadForEndpoints list will match payload
     * @param {number} src_ep - Source endpoint
     * @param {number} dest_ep - Destination endpoint
     * @returns {boolean} - True if payload should be skipped
     */
    shouldSkipPayload: function(src_ep, dest_ep) {
        // Check if this endpoint pair is in the "enable payload" list
        const enablePayload = this.config.enablePayloadForEndpoints.some(ep => 
            ep.src_ep === src_ep && ep.dest_ep === dest_ep
        );
        // Return true to skip if NOT in the enable list
        return !enablePayload;
    },

    /**
     * Select a message and update the counter (without opening analyzer)
     * @param {Object} messageData - Message data object from buffer
     */
    selectMessage: function(messageData) {
        console.log('[Frequency] Message selected from buffer:', messageData);

        // Check if we should skip payload matching for these endpoints
        const skipPayload = this.shouldSkipPayload(messageData.src_ep, messageData.dest_ep);
        
        // Create message signature for filtering using actual buffer data
        const signature = {
            src_add: messageData.src_add,
            dest_add: messageData.dest_add,
            src_ep: messageData.src_ep,
            dest_ep: messageData.dest_ep,
            payloadPrefix: skipPayload ? null : this.extractPayloadPrefix(messageData.payload)
        };

        if (skipPayload) {
            console.log('[Frequency] Skipping payload matching for endpoints', 
                       messageData.src_ep, '→', messageData.dest_ep, 
                       '(not in enablePayloadForEndpoints list)');
        } else {
            console.log('[Frequency] Enabling payload matching for endpoints', 
                       messageData.src_ep, '→', messageData.dest_ep);
        }

        console.log('[Frequency] Filter signature created:', signature);
        this.currentSignature = signature;
        this.currentMessageData = messageData;
        
        // Update buffer counter to show selection
        this.updateBufferCounter();
    },

    /**
     * Open frequency analyzer with currently selected message
     */
    openAnalyzer: function() {
        if (!this.currentMessageData || !this.currentMatchingMessages) {
            console.error('[Frequency] No message selected or no matches available');
            return;
        }

        console.log('[Frequency] Opening frequency analyzer...');
        this.showFrequencyAnalysis(this.currentMessageData, this.currentMatchingMessages);
    },

    /**
     * Show frequency analysis for selected message
     * @param {Object} messageData - Message data object
     * @param {Array} matchingMessages - Pre-filtered matching messages
     */
    showFrequencyAnalysis: function(messageData, matchingMessages) {
        console.log('[Frequency] Analyzing frequency for message:', messageData);

        // Use pre-filtered messages (already filtered when message was selected)
        const signature = this.currentSignature;
        
        console.log(`[Frequency] Found ${matchingMessages.length} matching messages`);
        
        if (matchingMessages.length < 2) {
            // If we found 0 or 1 matches, try without payload prefix (more lenient)
            if (signature.payloadPrefix) {
                console.log('[Frequency] Not enough matches with payload prefix, trying without it...');
                const lenientSignature = { ...signature, payloadPrefix: null };
                const lenientMatches = this.filterMatchingMessages(allMessages, lenientSignature);
                
                if (lenientMatches.length >= 2) {
                    console.log(`[Frequency] Found ${lenientMatches.length} matches without payload prefix (ignoring payload data)`);
                    // Automatically proceed with lenient matching
                    this.displayFrequencyModal(lenientSignature, lenientMatches, 
                        this.buildHistogram(this.calculateIntervals(lenientMatches)),
                        this.calculateStatistics(this.calculateIntervals(lenientMatches)));
                    return;
                }
            }
            
            // Show helpful error with suggestions
            let errorMsg = `Only ${matchingMessages.length} matching message(s) found in buffer.\n\n`;
            errorMsg += `Need at least 2 messages for frequency analysis.\n\n`;
            errorMsg += `💡 Tips:\n`;
            errorMsg += `• Click a message type that appears more frequently\n`;
            errorMsg += `• Increase "# of messages to display" to load more history\n`;
            errorMsg += `• Wait for more messages to arrive\n\n`;
            errorMsg += `Looking for: src=${signature.src_add}, dst=${signature.dest_add}, ep=${signature.src_ep}→${signature.dest_ep}`;
            
            alert(errorMsg);
            return;
        }

        // Calculate intervals
        const intervals = this.calculateIntervals(matchingMessages);
        
        // Generate histogram data
        const histogramData = this.buildHistogram(intervals);
        
        // Calculate statistics
        const stats = this.calculateStatistics(intervals);

        // Display in modal
        this.displayFrequencyModal(signature, matchingMessages, histogramData, stats);
    },

    /**
     * Get all messages from the buffer
     * @returns {Array} - Array of message objects
     */
    getAllMessagesFromBuffer: function() {
        try {
            if (typeof mainApp !== 'undefined' && typeof mainApp.getAllMessages === 'function') {
                const messages = mainApp.getAllMessages();
                console.log(`[Frequency] Retrieved ${messages.length} messages from buffer`);
                return messages;
            } else {
                console.error('[Frequency] mainApp.getAllMessages() not available');
                return [];
            }
        } catch (error) {
            console.error('[Frequency] Error getting messages:', error);
            return [];
        }
    },

    /**
     * Filter messages matching the signature
     * @param {Array} messages - All messages
     * @param {Object} signature - Message signature to match
     * @returns {Array} - Filtered messages sorted by timestamp
     */
    filterMatchingMessages: function(messages, signature) {
        if (window.MESSAGE_FREQUENCY_DEBUG) {
            console.log('[Frequency] Starting to filter messages. Signature:', signature);
            console.log('[Frequency] Sample message from buffer:', messages[0]);
        }
        
        const filtered = messages.filter((msg, index) => {
            const debugThis = window.MESSAGE_FREQUENCY_DEBUG && index < 3; // Debug first 3 messages if enabled
            
            if (debugThis) {
                console.log(`[Frequency] Checking message ${index}:`, {
                    src_add: msg.src_add,
                    dest_add: msg.dest_add,
                    src_ep: msg.src_ep,
                    dest_ep: msg.dest_ep,
                    payload_type: typeof msg.payload,
                    payload: msg.payload
                });
            }
            
            // Match source address (handle both number and string, and different formats)
            const msgSrcAdd = parseInt(msg.src_add);
            const sigSrcAdd = parseInt(signature.src_add);
            if (msgSrcAdd != sigSrcAdd) {
                if (debugThis) console.log(`  ❌ src_add mismatch: ${msg.src_add} (${msgSrcAdd}) != ${signature.src_add} (${sigSrcAdd})`);
                return false;
            }
            
            // Match destination address
            const msgDestAdd = parseInt(msg.dest_add);
            const sigDestAdd = parseInt(signature.dest_add);
            if (msgDestAdd != sigDestAdd) {
                if (debugThis) console.log(`  ❌ dest_add mismatch: ${msg.dest_add} (${msgDestAdd}) != ${signature.dest_add} (${sigDestAdd})`);
                return false;
            }
            
            // Match source endpoint
            if (msg.src_ep != signature.src_ep) {
                if (debugThis) console.log(`  ❌ src_ep mismatch: ${msg.src_ep} != ${signature.src_ep}`);
                return false;
            }
            
            // Match destination endpoint
            if (msg.dest_ep != signature.dest_ep) {
                if (debugThis) console.log(`  ❌ dest_ep mismatch: ${msg.dest_ep} != ${signature.dest_ep}`);
                return false;
            }
            
            // Match payload prefix (if specified in signature)
            if (signature.payloadPrefix) {
                try {
                    const msgPayloadPrefix = this.extractPayloadPrefix(msg.payload);
                    if (debugThis) {
                        console.log(`  Comparing payload prefixes:`);
                        console.log(`    Message: "${msgPayloadPrefix}"`);
                        console.log(`    Signature: "${signature.payloadPrefix}"`);
                    }
                    if (msgPayloadPrefix !== signature.payloadPrefix) {
                        if (debugThis) console.log(`  ❌ payload prefix mismatch`);
                        return false;
                    }
                } catch (error) {
                    console.warn('[Frequency] Error extracting payload prefix:', error);
                    // Continue without payload matching if there's an error
                }
            } else {
                if (debugThis) console.log(`  ℹ️ Payload matching skipped (positioning/tracking packet)`);
            }
            
            if (debugThis) console.log('  ✅ Message matches!');
            return true;
        });

        // Sort by timestamp
        filtered.sort((a, b) => {
            const timeA = this.parseTimestamp(a.rx_ts);
            const timeB = this.parseTimestamp(b.rx_ts);
            return timeA - timeB;
        });

        console.log(`[Frequency] Filtered ${filtered.length} matching messages from ${messages.length} total`);
        return filtered;
    },

    /**
     * Parse timestamp to milliseconds
     * @param {string|number} timestamp - Timestamp value
     * @returns {number} - Milliseconds
     */
    parseTimestamp: function(timestamp) {
        if (typeof timestamp === 'number') return timestamp;
        // Try to parse as date string
        const parsed = new Date(timestamp).getTime();
        return isNaN(parsed) ? 0 : parsed;
    },

    /**
     * Calculate time intervals between consecutive messages
     * @param {Array} messages - Sorted messages
     * @returns {Array} - Interval data (array of {intervalMs, prevMsg, currMsg} objects)
     */
    calculateIntervals: function(messages) {
        const intervals = [];
        
        for (let i = 1; i < messages.length; i++) {
            const prevTime = this.parseTimestamp(messages[i - 1].rx_ts);
            const currTime = this.parseTimestamp(messages[i].rx_ts);
            const intervalMs = currTime - prevTime;
            
            if (intervalMs >= 0) {  // Ignore negative intervals (clock issues)
                intervals.push({
                    intervalMs: intervalMs,
                    prevMsg: messages[i - 1],
                    currMsg: messages[i]
                });
            }
        }

        console.log(`[Frequency] Calculated ${intervals.length} intervals`);
        return intervals;
    },

    /**
     * Build histogram with individual bars for each unique interval
     * @param {Array} intervals - Array of interval objects {intervalMs, prevMsg, currMsg}
     * @returns {Object} - Histogram data {labels, counts, binEdges, messagePairs}
     */
    buildHistogram: function(intervals) {
        if (intervals.length === 0) {
            return { labels: [], counts: [], binEdges: [], messagePairs: {} };
        }

        // Convert to whole seconds (rounded) and group message pairs
        const bins = {};
        intervals.forEach(interval => {
            const seconds = Math.round(interval.intervalMs / 1000);
            if (!bins[seconds]) {
                bins[seconds] = [];
            }
            bins[seconds].push({
                prevMsg: interval.prevMsg,
                currMsg: interval.currMsg,
                intervalMs: interval.intervalMs
            });
        });
        
        // Get unique intervals and sort them
        const uniqueIntervals = Object.keys(bins)
            .map(Number)
            .sort((a, b) => a - b);
        
        // Create labels and count arrays
        const labels = uniqueIntervals.map(interval => `${interval}s`);
        const countsArray = uniqueIntervals.map(interval => bins[interval].length);
        
        // Create bin edges for reference
        const binEdges = uniqueIntervals.map(interval => ({
            start: interval,
            end: interval + 1
        }));
        
        // Create messagePairs lookup: key is interval in seconds, value is array of message pairs
        const messagePairs = {};
        uniqueIntervals.forEach(interval => {
            messagePairs[interval] = bins[interval];
        });
        
        return {
            labels: labels,
            counts: countsArray,
            binEdges: binEdges,
            messagePairs: messagePairs,
            binSize: 1  // Each bar is 1 second wide
        };
    },

    /**
     * Calculate statistics from intervals
     * @param {Array} intervals - Array of interval objects {intervalMs, prevMsg, currMsg}
     * @returns {Object} - Statistics
     */
    calculateStatistics: function(intervals) {
        if (intervals.length === 0) {
            return { count: 0 };
        }

        // Extract just the interval values in milliseconds
        const intervalValues = intervals.map(i => i.intervalMs);
        
        const sorted = [...intervalValues].sort((a, b) => a - b);
        const sum = intervalValues.reduce((a, b) => a + b, 0);
        const mean = sum / intervalValues.length;
        
        const median = sorted.length % 2 === 0
            ? (sorted[sorted.length / 2 - 1] + sorted[sorted.length / 2]) / 2
            : sorted[Math.floor(sorted.length / 2)];
        
        const variance = intervalValues.reduce((acc, val) => acc + Math.pow(val - mean, 2), 0) / intervalValues.length;
        const stdDev = Math.sqrt(variance);
        
        return {
            count: intervalValues.length,
            min: Math.min(...intervalValues),
            max: Math.max(...intervalValues),
            mean: mean,
            median: median,
            stdDev: stdDev
        };
    },

    /**
     * Create modal HTML structure
     */
    createModal: function() {
        // Check if modal already exists
        if (document.getElementById('message_frequency_modal')) {
            return;
        }

        const modal = document.createElement('div');
        modal.id = 'message_frequency_modal';
        modal.className = 'w3-modal';
        modal.style.display = 'none';
        modal.innerHTML = `
            <div class="w3-modal-content w3-card-4 w3-animate-zoom" style="max-width: 1000px;">
                <header class="w3-container wp-red" style="padding: 10px 16px;">
                    <span onclick="messageFrequencyAnalyzer.closeModal()" 
                          class="w3-button w3-display-topright w3-hover-red">&times;</span>
                    <h4 style="margin: 4px 0;"><i class="fa fa-bar-chart"></i> Message Histogram</h4>
                </header>
                
                <div class="w3-container" style="padding: 12px;">
                    <!-- Filter Criteria -->
                    <div id="frequency_filter_criteria" class="w3-panel w3-pale-blue w3-leftbar w3-border-blue" style="padding: 6px 12px; font-size: 12px;">
                        <!-- Populated dynamically -->
                    </div>
                    
                    <!-- Statistics -->
                    <div id="frequency_stats_summary" class="w3-row-padding" style="margin-top: 8px; font-size: 12px;">
                        <!-- Populated dynamically -->
                    </div>
                    
                    <!-- Chart -->
                    <div class="w3-container" style="height: 400px; position: relative; margin-top: 12px;">
                        <canvas id="frequency_histogram_chart"></canvas>
                    </div>
                    
                    <!-- Interval Details (expandable, populated on bar click) -->
                    <div id="frequency_interval_details" style="display: none; margin-top: 12px;">
                        <div class="w3-panel w3-pale-yellow w3-leftbar w3-border-orange" style="padding: 6px 12px; font-size: 12px;">
                            <p style="margin: 0 0 4px 0;">
                                <b id="interval_details_title">Interval Details</b>
                                <button onclick="messageFrequencyAnalyzer.closeIntervalDetails()" 
                                        class="w3-button w3-small w3-grey" 
                                        style="float: right; padding: 2px 8px; font-size: 11px; margin: -2px 0 0 0;">
                                    Close
                                </button>
                            </p>
                            <div id="interval_details_content" style="max-height: 300px; overflow-y: auto; font-family: monospace;">
                                <!-- Populated dynamically on bar click -->
                            </div>
                        </div>
                    </div>
                    
                    <!-- Actions -->
                    <div class="w3-container" style="padding: 8px 12px; margin-top: 8px;">
                        <button onclick="messageFrequencyAnalyzer.closeModal()" 
                                class="w3-button w3-grey w3-hover-dark-grey" style="font-size: 12px; padding: 6px 12px;">
                            <i class="fa fa-close"></i> Close
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        console.log('[Frequency] Modal created');
    },

    /**
     * Display frequency analysis in modal
     * @param {Object} signature - Message signature
     * @param {Array} messages - Matching messages
     * @param {Object} histogramData - Histogram data
     * @param {Object} stats - Statistics
     */
    displayFrequencyModal: function(signature, messages, histogramData, stats) {
        // Store histogram data for click handling
        this.currentHistogramData = histogramData;
        
        // Populate filter criteria (with timestamp range)
        this.populateFilterCriteria(signature, messages);
        
        // Populate statistics
        this.populateStatistics(messages.length, stats);
        
        // Render chart with interactive features
        this.renderChart(histogramData, stats);
        
        // Hide interval details section (in case it was open)
        this.closeIntervalDetails();
        
        // Show modal
        const modal = document.getElementById('message_frequency_modal');
        if (modal) {
            modal.style.display = 'block';
        }
    },

    /**
     * Populate filter criteria display
     * @param {Object} signature - Message signature
     * @param {Array} messages - Matching messages (already sorted by timestamp)
     */
    populateFilterCriteria: function(signature, messages) {
        const container = document.getElementById('frequency_filter_criteria');
        if (!container) return;

        // Extract first and last timestamps from sorted messages and format them
        let timestampInfo = '';
        if (messages && messages.length > 0) {
            const firstTimestamp = this.formatTimestampForDisplay(messages[0].rx_ts) || 'N/A';
            const lastTimestamp = this.formatTimestampForDisplay(messages[messages.length - 1].rx_ts) || 'N/A';
            timestampInfo = `<b>First:</b> ${firstTimestamp} → <b>Last:</b> ${lastTimestamp}<br>`;
        }

        container.innerHTML = `
            <p style="margin: 0; font-weight: bold;">Filter Criteria</p>
            <p style="margin: 3px 0 0 0;">
                <b>Source Address:</b> ${this.formatAddress(signature.src_add)}<br>
                <b>Destination Address:</b> ${this.formatAddress(signature.dest_add)}<br>
                <b>Endpoints:</b> ${signature.src_ep} → ${signature.dest_ep}<br>
                <b>Payload Prefix:</b> ${signature.payloadPrefix || 'N/A'}<br>
                ${timestampInfo}
            </p>
        `;
    },

    /**
     * Format address for display
     * @param {number} address - Address value
     * @returns {string} - Formatted address
     */
    formatAddress: function(address) {
        if (address === undefined || address === null) return 'N/A';
        return `0x${address.toString(16).toUpperCase().padStart(8, '0')} (${address})`;
    },

    /**
     * Populate statistics display
     * @param {number} messageCount - Total matching messages
     * @param {Object} stats - Statistics object
     */
    populateStatistics: function(messageCount, stats) {
        const container = document.getElementById('frequency_stats_summary');
        if (!container) return;

        const formatMs = (ms) => {
            if (ms >= 1000) {
                return `${(ms / 1000).toFixed(2)}s`;
            } else {
                return `${ms.toFixed(0)}ms`;
            }
        };

        container.innerHTML = `
            <div class="w3-third">
                <div class="w3-container w3-pale-green" style="padding: 6px 8px;">
                    <p style="margin: 0; font-weight: bold; font-size: 11px;">Messages</p>
                    <p style="margin: 2px 0; font-size: 14px; font-weight: bold;">${messageCount}</p>
                    <p style="margin: 0; font-size: 10px; color: #666;">Intervals: ${stats.count}</p>
                </div>
            </div>
            <div class="w3-third">
                <div class="w3-container w3-pale-yellow" style="padding: 6px 8px;">
                    <p style="margin: 0; font-weight: bold; font-size: 11px;">Mean</p>
                    <p style="margin: 2px 0; font-size: 14px; font-weight: bold;">${formatMs(stats.mean)}</p>
                    <p style="margin: 0; font-size: 10px; color: #666;">Std Dev: ${formatMs(stats.stdDev)}</p>
                </div>
            </div>
            <div class="w3-third">
                <div class="w3-container w3-pale-blue" style="padding: 6px 8px;">
                    <p style="margin: 0; font-weight: bold; font-size: 11px;">Median</p>
                    <p style="margin: 2px 0; font-size: 14px; font-weight: bold;">${formatMs(stats.median)}</p>
                    <p style="margin: 0; font-size: 10px; color: #666;">Range: ${formatMs(stats.min)} - ${formatMs(stats.max)}</p>
                </div>
            </div>
        `;
    },

    /**
     * Render histogram chart using Chart.js
     * @param {Object} histogramData - Histogram data
     * @param {Object} stats - Statistics
     */
    renderChart: function(histogramData, stats) {
        const canvas = document.getElementById('frequency_histogram_chart');
        if (!canvas) {
            console.error('[Frequency] Chart canvas not found');
            return;
        }

        // Destroy existing chart
        if (this.chartInstance) {
            this.chartInstance.destroy();
        }

        const ctx = canvas.getContext('2d');
        
        // Convert stats to seconds for display
        const meanSeconds = stats.mean / 1000;
        const medianSeconds = stats.median / 1000;

        this.chartInstance = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: histogramData.labels,
                datasets: [{
                    label: 'Number of Messages',
                    data: histogramData.counts,
                    backgroundColor: this.config.chartColors.barFill,
                    borderColor: this.config.chartColors.barBorder,
                    borderWidth: 1,
                    barPercentage: 0.9,  // Make bars thinner
                    categoryPercentage: 0.8  // Reduce spacing
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                onClick: (event, activeElements) => {
                    // Handle bar clicks
                    if (activeElements && activeElements.length > 0) {
                        const index = activeElements[0].index;
                        const label = histogramData.labels[index];
                        const intervalSeconds = parseInt(label);  // Extract seconds from "300s" format
                        this.showIntervalDetails(intervalSeconds);
                    }
                },
                plugins: {
                    legend: {
                        display: true,
                        position: 'top'
                    },
                    title: {
                        display: true,
                        text: 'Time Interval Distribution (click bars for details)'
                    },
                    tooltip: {
                        callbacks: {
                            title: function(context) {
                                return `Interval: ${context[0].label}`;
                            },
                            label: function(context) {
                                const count = context.parsed.y;
                                return `${count} message ${count === 1 ? 'pair' : 'pairs'}`;
                            },
                            afterLabel: function(context) {
                                return 'Click to see timestamps';
                            }
                        }
                    },
                    annotation: {
                        annotations: {
                            meanLine: {
                                type: 'line',
                                xMin: meanSeconds,
                                xMax: meanSeconds,
                                borderColor: this.config.chartColors.meanLine,
                                borderWidth: 2,
                                borderDash: [5, 5],
                                label: {
                                    enabled: true,
                                    content: `Mean: ${(meanSeconds).toFixed(2)}s`,
                                    position: 'start'
                                }
                            },
                            medianLine: {
                                type: 'line',
                                xMin: medianSeconds,
                                xMax: medianSeconds,
                                borderColor: this.config.chartColors.medianLine,
                                borderWidth: 2,
                                borderDash: [10, 5],
                                label: {
                                    enabled: true,
                                    content: `Median: ${(medianSeconds).toFixed(2)}s`,
                                    position: 'end'
                                }
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1,
                            precision: 0
                        },
                        title: {
                            display: true,
                            text: 'Number of Messages'
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'Time Interval (seconds)'
                        },
                        ticks: {
                            autoSkip: false,  // Show all labels
                            maxRotation: 45,
                            minRotation: 0
                        }
                    }
                }
            }
        });

        console.log('[Frequency] Chart rendered');
    },

    /**
     * Close the modal
     */
    closeModal: function() {
        const modal = document.getElementById('message_frequency_modal');
        if (modal) {
            modal.style.display = 'none';
        }
        
        // Remove selection highlight
        const selected = document.querySelector('.frequency-selected');
        if (selected) {
            selected.classList.remove('frequency-selected');
        }
        
        // Clear current signature and stored data
        this.currentSignature = null;
        this.currentMessageData = null;
        this.currentMatchingMessages = null;
        this.currentHistogramData = null;
        this.updateBufferCounter();
    },

    /**
     * Show details for a specific interval bin
     * @param {number} intervalSeconds - The interval in seconds
     */
    showIntervalDetails: function(intervalSeconds) {
        if (!this.currentHistogramData || !this.currentHistogramData.messagePairs) {
            console.warn('[Frequency] No histogram data available');
            return;
        }

        const messagePairs = this.currentHistogramData.messagePairs[intervalSeconds];
        if (!messagePairs || messagePairs.length === 0) {
            console.warn(`[Frequency] No message pairs found for interval ${intervalSeconds}s`);
            return;
        }

        // Update title
        const titleEl = document.getElementById('interval_details_title');
        if (titleEl) {
            titleEl.textContent = `${intervalSeconds}s Interval - ${messagePairs.length} message ${messagePairs.length === 1 ? 'pair' : 'pairs'}`;
        }

        // Build content HTML
        const contentEl = document.getElementById('interval_details_content');
        if (!contentEl) return;

        let html = '<table style="width: 100%; font-size: 11px; border-collapse: collapse;">';
        html += '<tr style="background-color: #f0f0f0; font-weight: bold;">';
        html += '<th style="padding: 4px; text-align: left; border-bottom: 1px solid #ccc;">#</th>';
        html += '<th style="padding: 4px; text-align: left; border-bottom: 1px solid #ccc;">First Message</th>';
        html += '<th style="padding: 4px; text-align: left; border-bottom: 1px solid #ccc;">Second Message</th>';
        html += '<th style="padding: 4px; text-align: right; border-bottom: 1px solid #ccc;">Interval</th>';
        html += '</tr>';

        messagePairs.forEach((pair, index) => {
            const prevTimestamp = this.formatTimestampForDisplay(pair.prevMsg.rx_ts);
            const currTimestamp = this.formatTimestampForDisplay(pair.currMsg.rx_ts);
            const intervalMs = pair.intervalMs.toFixed(0);
            const intervalDisplay = intervalMs >= 1000 
                ? `${(intervalMs / 1000).toFixed(2)}s` 
                : `${intervalMs}ms`;

            html += `<tr style="border-bottom: 1px solid #eee;">`;
            html += `<td style="padding: 4px;">${index + 1}</td>`;
            html += `<td style="padding: 4px;">${prevTimestamp}</td>`;
            html += `<td style="padding: 4px;">${currTimestamp}</td>`;
            html += `<td style="padding: 4px; text-align: right;">${intervalDisplay}</td>`;
            html += '</tr>';
        });

        html += '</table>';
        contentEl.innerHTML = html;

        // Show the details section
        const detailsContainer = document.getElementById('frequency_interval_details');
        if (detailsContainer) {
            detailsContainer.style.display = 'block';
        }

        console.log(`[Frequency] Showing details for ${intervalSeconds}s interval (${messagePairs.length} pairs)`);
    },

    /**
     * Close the interval details section
     */
    closeIntervalDetails: function() {
        const detailsContainer = document.getElementById('frequency_interval_details');
        if (detailsContainer) {
            detailsContainer.style.display = 'none';
        }
    }
};

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    console.log('[Frequency] DOM loaded, initializing in 1000ms...');
    setTimeout(() => {
        console.log('[Frequency] Starting initialization...');
        messageFrequencyAnalyzer.initialize();
    }, 1000);
    
    // Also try again after 3 seconds in case messages load later
    setTimeout(() => {
        console.log('[Frequency] Re-attempting initialization (in case messages loaded late)...');
        messageFrequencyAnalyzer.addMessageClickHandlers();
    }, 3000);
});

// Close modal when clicking outside
window.addEventListener('click', function(event) {
    const modal = document.getElementById('message_frequency_modal');
    if (event.target === modal) {
        messageFrequencyAnalyzer.closeModal();
    }
});

// Helper function to show most frequent message types in buffer
window.showFrequentMessages = function() {
    console.log('[Frequency] Analyzing message types in buffer...');
    
    if (typeof mainApp === 'undefined' || typeof mainApp.getAllMessages !== 'function') {
        console.error('mainApp.getAllMessages() not available');
        return;
    }
    
    const messages = mainApp.getAllMessages();
    const counts = {};
    
    // Count message types by src_add, dest_add, src_ep, dest_ep
    messages.forEach(msg => {
        const key = `${msg.src_add}→${msg.dest_add} (ep:${msg.src_ep}→${msg.dest_ep})`;
        counts[key] = (counts[key] || 0) + 1;
    });
    
    // Sort by frequency
    const sorted = Object.entries(counts)
        .sort((a, b) => b[1] - a[1])
        .slice(0, 10); // Top 10
    
    console.log('%c━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━', 'color: #4CAF50');
    console.log('%cMost Frequent Message Types in Buffer:', 'color: #4CAF50; font-weight: bold');
    console.log('%c━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━', 'color: #4CAF50');
    
    sorted.forEach(([key, count], index) => {
        console.log(`${index + 1}. ${key} - ${count} messages`);
    });
    
    console.log('%c━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━', 'color: #4CAF50');
    console.log('💡 Click a message matching one of these types for best results!');
};

// Helper function for manual testing - analyze the first message in the list
window.testFrequencyAnalyzer = function() {
    console.log('[Frequency Test] Manually triggering frequency analyzer...');
    
    const messageList = document.querySelector('#message_list .list, #message_list ul, #message_list');
    if (!messageList) {
        console.error('[Frequency Test] Could not find message list');
        console.log('[Frequency Test] Available elements:', document.querySelectorAll('[id*="message"]'));
        return;
    }
    
    console.log('[Frequency Test] Message list found:', messageList);
    
    // Try to find first message using various selectors
    let firstMessage = messageList.querySelector('li');
    if (!firstMessage) {
        firstMessage = messageList.querySelector('p');
    }
    if (!firstMessage) {
        firstMessage = messageList.querySelector('div');
    }
    if (!firstMessage) {
        firstMessage = messageList.firstElementChild;
    }
    
    if (!firstMessage) {
        console.error('[Frequency Test] No messages found in list');
        console.log('[Frequency Test] Message list children:', messageList.children);
        return;
    }
    
    console.log('[Frequency Test] Found first message:', firstMessage);
    console.log('[Frequency Test] Message HTML:', firstMessage.outerHTML.substring(0, 200));
    
    // Simulate click
    firstMessage.click();
};

// Log helper message on load
setTimeout(() => {
    console.log('%c━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━', 'color: #4CAF50');
    console.log('%cMessage Frequency Analyzer - Loaded', 'color: #4CAF50; font-weight: bold; font-size: 16px');
    console.log('%c━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━', 'color: #4CAF50');
    console.log('%cHow to use:', 'font-weight: bold');
    console.log('1. Click any message in the list');
    console.log('2. Or run: testFrequencyAnalyzer()');
    console.log('%cHelpers:', 'font-weight: bold');
    console.log('- Show frequent message types: showFrequentMessages()');
    console.log('- Enable debug: MESSAGE_FREQUENCY_DEBUG = true');
    console.log('- Check status: messageFrequencyAnalyzer');
    console.log('%c━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━', 'color: #4CAF50');
}, 2000);

