/**
 * ui_messages_evertag.js
 *
 * Extends ui_messages.js to add EverTag message parsing and display.
 * - Adds UI toggle for EverTag parsing.
 * - Uses MutationObserver to update message display with parsed data.
 * - Debug options available via global flags.
 * - See README.md for extension patterns.
 */

// Global flag to enable/disable all tag parsing debug logs
window.TAG_PARSING_DEBUG = false;
// Global flag to enable/disable EverTag App Config debug logs
window.EVERTAG_APPCONFIG_DEBUG = false;

// On load, print debug help for available debug variables
// This helps users discover and enable debug tracing easily
if (typeof window !== 'undefined' && window.console) {
    console.log('%c[EverTag Debug Options]', 'color: #b8860b; font-weight: bold;');
    console.log('To enable EverTag message parsing debug logs, run:');
    console.log('   TAG_PARSING_DEBUG = true;');
    console.log('To enable EverTag App Config debug logs, run:');
    console.log('   EVERTAG_APPCONFIG_DEBUG = true;');
    console.log('Set either variable to false to disable the corresponding debug output.');
    console.log('You can copy/paste the above commands into your browser console.');
}

// Global flag to track if tag parsing is enabled
let isTagParsingEnabled = false;

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize the EverTag parsing functionality
    setTimeout(initializeEverTagParsing, 600); // Delay to ensure uiMessages is fully loaded
});

/**
 * Initialize the EverTag parsing functionality
 */
function initializeEverTagParsing() {
    if (!window.uiMessages) {
        console.error('uiMessages not found. EverTag parsing cannot be initialized.');
        return;
    }
    
    // Add the onParseTagDataChanged function to uiMessages
    uiMessages.onParseTagDataChanged = function(checkbox) {
        isTagParsingEnabled = checkbox.checked;
        // Re-parse only if enabled, otherwise remove parsed data
        if (isTagParsingEnabled) {
            observeEverTagMessages();
            // Process all current messages
            const allLiElements = document.querySelectorAll('#message_list li');
            for (let i = 0; i < allLiElements.length; i++) {
                updateSingleMessageDisplay(allLiElements[i]);
            }
        } else {
            // Remove all parsed data
            const parsedElements = document.querySelectorAll('.evertag-parsed-data');
            parsedElements.forEach(el => el.remove());
        }
    };
    
    // Debug: Log that EverTag parsing was initialized
    // This helps confirm that the EverTag extension is active
    if (window.TAG_PARSING_DEBUG) {
        console.log('EverTag parsing initialized successfully');
    }
}

/**
 * Add a robust MutationObserver-based post-processing approach
 */
function observeEverTagMessages() {
    const messageListContainer = document.getElementById('message_list');
    if (!messageListContainer) return;

    // Disconnect any previous observer
    if (window.evertagMessageObserver) {
        window.evertagMessageObserver.disconnect();
    }

    window.evertagMessageObserver = new MutationObserver(function(mutations) {
        if (!isTagParsingEnabled) return;
        for (const mutation of mutations) {
            for (const node of mutation.addedNodes) {
                if (node.nodeType === 1 && !node.querySelector('.evertag-parsed-data')) {
                    updateSingleMessageDisplay(node);
                }
            }
        }
    });

    window.evertagMessageObserver.observe(messageListContainer, {
        childList: true,
        subtree: true
    });
}

/**
 * Update the display of a single message to include/remove tag parsing
 * @param {HTMLElement} messageItem - The message list item element
 */
function updateSingleMessageDisplay(messageItem) {
    if (!isIndividualMessage(messageItem)) return;

    let parsedDataElement = messageItem.querySelector('.evertag-parsed-data');
    const payloadText = findPayloadInMessage(messageItem);

    // Extract src_ep and dst_ep from the DOM if available
    let srcEp = null, dstEp = null;
    const srcEpEl = messageItem.querySelector('.src_ep');
    const dstEpEl = messageItem.querySelector('.dest_ep');
    if (srcEpEl && dstEpEl) {
        srcEp = parseInt(srcEpEl.textContent.trim());
        dstEp = parseInt(dstEpEl.textContent.trim());
    }

    // Debug: Log what we are about to parse
    // Shows the payload, source/destination endpoints, and parsing flag
    // Useful to verify what data is being processed for each message
    if (window.TAG_PARSING_DEBUG) {
        console.log('[EverTag DEBUG] updateSingleMessageDisplay:', {payloadText, srcEp, dstEp, isTagParsingEnabled});
    }

    if (isTagParsingEnabled && payloadText) {
        // Only update if payload changed or parsed row doesn't exist
        const lastPayload = parsedDataElement ? parsedDataElement.getAttribute('data-last-payload') : null;
        if (!parsedDataElement || lastPayload !== payloadText) {
            const parsedData = parseEverTagData(payloadText, srcEp, dstEp);
            // Debug: Log the parsed data result
            // Shows the output of the EverTag parser for this message
            // Helps verify if parsing logic is working as expected
            if (window.TAG_PARSING_DEBUG) {
                console.log('[EverTag DEBUG] Parsed data:', parsedData);
            }
            if (parsedData !== null) {
                if (!parsedDataElement) {
                    parsedDataElement = document.createElement('div');
                    parsedDataElement.className = 'evertag-parsed-data';
                    parsedDataElement.style.cssText = `
                        margin-top: 8px;
                        clear: both;
                        width: 100%;
                        display: block;
                    `;
                    messageItem.appendChild(parsedDataElement);
                }
                parsedDataElement.setAttribute('data-last-payload', payloadText);
                parsedDataElement.innerHTML = `<span class="terminal_field decoded_data">${parsedData}</span>`;
            } else if (parsedDataElement) {
                // Remove if not relevant anymore
                parsedDataElement.remove();
            }
        }
    } else if (parsedDataElement) {
        parsedDataElement.remove();
    }
}

/**
 * Check if an element represents an individual message (not a container of messages)
 * @param {HTMLElement} element - Element to check
 * @returns {boolean} - True if it's likely an individual message
 */
function isIndividualMessage(element) {
    // Count how many hex payloads are directly in this element
    const hexElements = Array.from(element.querySelectorAll('*')).filter(el => {
        const text = (el.textContent || '').trim();
        return isHexPayload(text);
    });
    
    // If there's exactly one hex payload, it's likely an individual message
    // If there are multiple, it's likely a container of messages
    const hasExactlyOnePayload = hexElements.length === 1;
    
    // Also check if it has a reasonable message structure (some fields but not too many)
    const terminalFields = element.querySelectorAll('.terminal_field, [class*="gw_id"], [class*="src_"], [class*="dest_"]');
    const hasReasonableFieldCount = terminalFields.length >= 1 && terminalFields.length <= 20;
    
    return hasExactlyOnePayload && hasReasonableFieldCount;
}

/**
 * Find payload data within a specific message item
 * @param {HTMLElement} messageItem - The message container element
 * @returns {string|null} - The payload text or null if not found
 */
function findPayloadInMessage(messageItem) {
    // Look for text that appears to be hex payload data
    const allTextElements = messageItem.querySelectorAll('*');
    
    for (let element of allTextElements) {
        const text = (element.textContent || '').trim();
        
        // Check if this looks like hex payload data
        if (isHexPayload(text)) {
            return text;
        }
    }
    
    return null;
}

/**
 * Check if text looks like hex payload data
 * @param {string} text - Text to check
 * @returns {boolean} - True if it looks like hex payload
 */
function isHexPayload(text) {
    // Must be at least 4 characters and contain hex-like patterns
    if (text.length < 4) return false;
    
    // Check if it's mostly hex characters and spaces
    const hexPattern = /^[\da-fA-F\s]+$/;
    if (!hexPattern.test(text)) return false;
    
    // Should have spaces between hex pairs typically
    const hexPairs = text.split(/\s+/).filter(s => s.length > 0);
    
    // Check if we have reasonable hex pairs (1-2 characters each)
    if (hexPairs.length === 0) return false;
    
    for (let pair of hexPairs) {
        if (pair.length > 2 || !/^[\da-fA-F]+$/.test(pair)) {
            return false;
        }
    }
    
    return true;
}

/**
 * Parse EverTag data from payload
 * @param {string} payloadHex - The payload in hex format
 * @param {number} src_ep - Source endpoint
 * @param {number} dst_ep - Destination endpoint
 * @returns {string} - Parsed data string
 */
function parseEverTagData(payloadHex, src_ep, dst_ep) {
    // Debug: Log entry to parsing function
    // Shows the input payload and endpoints for each parse attempt
    // Useful for tracing parsing issues or unexpected data
    if (window.TAG_PARSING_DEBUG) {
        console.log('[EverTag DEBUG] parseEverTagData called:', {payloadHex, src_ep, dst_ep});
    }
    
    // Wirepas Positioning (EP 238/238) - keep this as it's positioning data
    if (src_ep === 238 && dst_ep === 238) {
        if (!payloadHex) return 'Positioning: No payload';
        const hexPairs = payloadHex.trim().split(/\s+/).map(s => s.toLowerCase());
        if (hexPairs.length < 2) return 'Positioning: Payload too short';

        // Parse sequence number (2 bytes, little endian)
        let seq = null;
        try {
            const seqLow = parseInt(hexPairs[0], 16);
            const seqHigh = parseInt(hexPairs[1], 16);
            seq = (seqHigh << 8) | seqLow;
        } catch (e) {
            return 'Positioning: Invalid sequence number';
        }

        // Parse TLV records
        let i = 2;
        let fields = [];
        while (i + 1 < hexPairs.length) {
            const type = parseInt(hexPairs[i], 16);
            const len = parseInt(hexPairs[i + 1], 16);
            if (isNaN(type) || isNaN(len) || i + 2 + len > hexPairs.length) {
                fields.push('Malformed TLV');
                i += 1;
                continue;
            }
            const valueBytes = hexPairs.slice(i + 2, i + 2 + len);
            // Anchor RSSI (0xF5) or Tag RSSI (0x05)
            if ((type === 0xF5 || type === 0x05) && len >= 5) {
                // Iterate over each 5-byte anchor record
                for (let anchorIdx = 0; anchorIdx + 5 <= valueBytes.length; anchorIdx += 5) {
                    const addrBytes = valueBytes.slice(anchorIdx, anchorIdx + 4);
                    // Little-endian to integer
                    const addrInt = addrBytes.reduceRight((acc, b) => (acc << 8) | parseInt(b, 16), 0);
                    const addr = addrInt.toString(16).padStart(8, '0').toUpperCase();
                    const rssiByte = parseInt(valueBytes[anchorIdx + 4], 16);
                    const rssiDbm = (rssiByte * -0.5).toFixed(1);
                    const label = type === 0xF5 ? 'Anchor' : 'Tag';
                    fields.push(`${label}: 0x${addr} RSSI: ${rssiDbm} dBm`);
                }
            }
            // Battery voltage (0x04)
            else if (type === 0x04 && len === 2) {
                const vLow = parseInt(valueBytes[0], 16);
                const vHigh = parseInt(valueBytes[1], 16);
                const voltage = ((vHigh << 8) | vLow);
                fields.push(`Battery: ${voltage} mV`);
            }
            // Tag operation info (0x06)
            else if (type === 0x06 && len >= 1) {
                const modeByte = parseInt(valueBytes[0], 16);
                let modeStr = 'Unknown';
                if (modeByte === 0x00) modeStr = 'NRLS';
                fields.push(`Mode: ${modeStr}`);
            }
            // Unknown type
            else {
                fields.push(`Unknown type 0x${type.toString(16).toUpperCase()}`);
            }
            const prevI = i;
            i += 2 + len;
        }
        let result = `Positioning: Seq: ${seq}`;
        if (fields.length > 0) {
            result += ' | ' + fields.join(' | ');
        }
        return result;
    }
    
    // Diagnostics (EP 247/255) - keep this as it's diagnostics data
    if (src_ep === 247 && dst_ep === 255) {
        return 'Diagnostics information';
    }

    // ONLY parse as EverTag data if it's on the correct EverTag endpoints
    const isEverTagLegacy = (src_ep === 130 && dst_ep === 130);
    const isEverTagNew = (src_ep === 160 && dst_ep === 107);
    
    if (!isEverTagLegacy && !isEverTagNew) {
        // Not an EverTag endpoint - return null to indicate no parsing should be done
        return null;
    }

    if (!payloadHex) {
        const prefix = isEverTagLegacy ? 'EverTag Legacy: ' : 'EverTag: ';
        return `${prefix}No payload`;
    }

    const hexPairs = payloadHex.trim().split(/\s+/).map(s => s.toLowerCase());
    if (hexPairs.length < 1) {
        const prefix = isEverTagLegacy ? 'EverTag Legacy: ' : 'EverTag: ';
        return `${prefix}Empty payload`;
    }

    // Use these for all event/config parsing
    const firstByte = parseInt(hexPairs[0], 16);
    const prefix = isEverTagLegacy ? 'EverTag Legacy: ' : 'EverTag: ';

    // EverTag Event Frame parsing (per Tag_events_over_Wirepas_Mesh.md)
    if (firstByte === 0x0F && hexPairs.length >= 5) {
        // Parse 4-byte hardware code as little-endian 32-bit value and display as decimal
        const hwBytes = hexPairs.slice(1, 5);
        // Convert little-endian bytes to 32-bit integer
        const hwCodeValue = parseInt(hwBytes[0], 16) |
                           (parseInt(hwBytes[1], 16) << 8) |
                           (parseInt(hwBytes[2], 16) << 16) |
                           (parseInt(hwBytes[3], 16) << 24);
        
        // Display as decimal with hex in parentheses for reference
        const hwCodeHex = hwBytes.join('').toUpperCase();
        return `${prefix}Hardware code: ${hwCodeValue} (0x${hwCodeHex})`;
    }
    if (firstByte === 0x10 && hexPairs.length >= 5) {
        const major = parseInt(hexPairs[1], 16);
        const minor = parseInt(hexPairs[2], 16);
        const maint = parseInt(hexPairs[3], 16);
        const dev = parseInt(hexPairs[4], 16);
        return `${prefix}Firmware version: v${major}.${minor}.${maint}.${dev}`;
    }
    if (firstByte === 0x12 && hexPairs.length >= 2) {
        const battery = parseInt(hexPairs[1], 16);
        return `${prefix}Battery level: ${battery}%`;
    }
    if (firstByte === 0x13 && hexPairs.length >= 14) {
        // 0x13 [temp] [movement:4B] [uptime:4B] [moving:4B]
        const tempRaw = parseInt(hexPairs[1], 16);
        const tempC = (tempRaw / 2 - 40).toFixed(1);
        // Movement status (signed 32-bit LE)
        const movBytes = hexPairs.slice(2, 6).reverse();
        let movVal = parseInt(movBytes.join(''), 16);
        if (movVal & 0x80000000) movVal = movVal - 0x100000000;
        // Uptime (unsigned 32-bit LE)
        const uptime = parseInt(hexPairs.slice(6, 10).reverse().join(''), 16);
        // Moving time (unsigned 32-bit LE)
        const moving = parseInt(hexPairs.slice(10, 14).reverse().join(''), 16);
        return `${prefix}BLoC: Temp: ${tempC}°C, Movement: ${movVal} s, Uptime: ${uptime} s, Moving: ${moving} s`;
    }
    if (firstByte === 0x14 && hexPairs.length >= 2) {
        const status = parseInt(hexPairs[1], 16);
        const alert = (status & 0x01) ? 'Alert (button),' : '';
	const tamper = (status & 0x02) ? 'Tamper,' : '';
	const ext_power = (status & 0x04) ? 'External power,' : '';
        return `${prefix}Status: ${alert} ${tamper} ${ext_power}`;
    }
    if (firstByte === 0x15 && hexPairs.length >= 3) {
        // 0x15: High resolution temperature (0.01°C), 16-bit signed int, little-endian
        const raw = (parseInt(hexPairs[1], 16)) | (parseInt(hexPairs[2], 16) << 8);
        // Convert to signed 16-bit integer
        const signed = raw & 0x8000 ? raw - 0x10000 : raw;
        const temp = (signed * 0.01).toFixed(2);
        return `${prefix}High-res temp: ${temp}°C`;
    }
    if (firstByte === 0x16 && hexPairs.length >= 3) {
        // 0x16 is now Ambient light (lux)
        const val = (parseInt(hexPairs[1], 16)) | (parseInt(hexPairs[2], 16) << 8);
        return `${prefix}Ambient light: ${val} lux`;
    }
    if (firstByte === 0x17 && hexPairs.length >= 3) {
        // 0x17 is now Ambient sound (dB)
        const val = (parseInt(hexPairs[1], 16)) | (parseInt(hexPairs[2], 16) << 8);
        const sound = (val * 0.01).toFixed(1);
        return `${prefix}Ambient sound: ${sound} dB`;
    }
    if (firstByte === 0x18 && hexPairs.length >= 3) {
        // 0x18 is vitals (2B flags, 2B update (1B start, 1B end), 2B stack (1B offline, 1B online), 2B accel (1B Irq, 1B status), 1B reset_reason, 1B reboots)
	const flags = parseInt(hexPairs[1], 16) | (parseInt(hexPairs[2], 16) << 8);
	const updateStart = parseInt(hexPairs[3], 16);
	const updateEnd = parseInt(hexPairs[4], 16);
	const stackOffline = parseInt(hexPairs[5], 16);
	const stackOnline = parseInt(hexPairs[6], 16);
	const accelIrq = parseInt(hexPairs[7], 16);
	const accelStatus = parseInt(hexPairs[8], 16);
	const resetReason = parseInt(hexPairs[9], 16);
	const reboots = parseInt(hexPairs[10], 16);
	return `${prefix}Vitals: Flags: ${flags}, Update: ${updateStart} - ${updateEnd}, Stack: ${stackOffline} - ${stackOnline}, Accel: ${accelIrq} - ${accelStatus}, Reset: ${resetReason}, Reboots: ${reboots}`;
    }
    if(firstByte === 0x19 && hexPairs.length >= 3) {
	const outputFlags = parseInt(hexPairs[1], 16);
	const inputFlags = parseInt(hexPairs[2], 16);
	const out1 = (outputFlags & 0x01) ? 'Active' : 'Inactive';
	const out2 = (outputFlags & 0x02) ? 'Active' : 'Inactive';
	const inpState = {0: 'Floating', 1: 'Low', 2: 'High', 3: 'Invalid'};
	const in1 = inpState[inputFlags & 0x03] || 'Unknown';
	const in2 = inpState[(inputFlags >> 4) & 0x03] || 'Unknown';
	return `${prefix}Digital I/O: Out1: ${out1}, Out2: ${out2}, In1: ${in1}, In2: ${in2}`;
    }
    // Fallback for unknown event codes
    if (firstByte >= 0x0F && firstByte <= 0x17) {
        return `${prefix}Event 0x${firstByte.toString(16).toUpperCase()} - Data: ${hexPairs.slice(1).join(' ').toUpperCase()}`;
    }

    // EverTag Configuration Protocol parsing
    if (firstByte === 0xB0 && hexPairs.length >= 3) {
        const seq = parseInt(hexPairs[1], 16);
        const length = parseInt(hexPairs[2], 16);
        
        if (hexPairs.length >= 4) {
            const msgType = parseInt(hexPairs[3], 16);
            const isTag = (msgType & 0x80) !== 0;
            const isAnchor = (msgType & 0x40) !== 0;
            const addrCount = msgType & 0x0F;
            
            let targetStr = '';
            if (isTag) targetStr = 'Tags';
            else if (isAnchor) targetStr = 'Anchors';
            else if (addrCount === 0) targetStr = 'Unicast';
            else targetStr = `Multicast (${addrCount})`;
            
            // Debug: Print App Config details if enabled
            // Shows the raw payload, endpoints, and parsed config info
            if (window.EVERTAG_APPCONFIG_DEBUG) {
                console.log('[EverTag AppConfig DEBUG]', {
                    payloadHex,
                    src_ep,
                    dst_ep,
                    seq,
                    length,
                    msgType,
                    targetStr,
                    hexPairs: hexPairs.join(' ')
                });
            }
            return `${prefix}Config - Seq: ${seq}, Target: ${targetStr}, Length: ${length}`;
        }
        
        // Debug: Print App Config details for short config if enabled
        if (window.EVERTAG_APPCONFIG_DEBUG) {
            console.log('[EverTag AppConfig DEBUG]', {
                payloadHex,
                src_ep,
                dst_ep,
                seq,
                length,
                hexPairs: hexPairs.join(' ')
            });
        }
        return `${prefix}Config - Seq: ${seq}, Length: ${length}`;
    }

    // EverTag Command Responses (0x01-0x1C range)
    if (firstByte >= 0x01 && firstByte <= 0x1C) {
        // Map command IDs to names
        const commandNames = {
            0x01: 'CBeacon advertisement interval',
            0x02: 'CBeacon TX power',
            0x03: 'Movement timeout',
            0x04: 'Movement threshold',
            0x05: 'Movement positioning interval',
            0x06: 'Idle positioning interval',
            0x07: 'Alert positioning interval',
            0x08: 'Alert timeout',
            // 0x09: 'LED intensity', // OBSOLETED - LED intensity no longer supported
            0x0A: 'Start LED pattern',
            0x0B: 'Deactivate tag',
            0x0C: 'Reset tag configuration',
            0x0D: 'Enable CBeacon adverts',
            0x0E: 'Disable CBeacon adverts',
            0x0F: 'Get hardware info',
            0x10: 'Get firmware info',
            // 0x11: 'Get sensor diagnostics', // OBSOLETED - Get sensor diagnostics no longer supported
            0x12: 'Get battery level',
            0x13: 'BloC report interval',
            0x14: 'Button shut-off timeout',
            0x15: 'iBeacon mode',
            0x16: 'iBeacon UUID',
            0x17: 'iBeacon major/minor',
            0x18: 'NFC PIN',
            0x19: 'Enable anchor tamper detection',
            0x1A: 'Disable anchor tamper detection',
            0x1B: 'EverTag-T temperature sample interval',
            0x1C: 'EverTag-SL sound/light sample interval',
            0x1D: 'EverTag-IO Output Control'
        };
        
        const commandName = commandNames[firstByte] || `Unknown command 0x${firstByte.toString(16).toUpperCase()}`;
        
        // Show additional data if present
        if (hexPairs.length > 1) {
            const dataBytes = hexPairs.slice(1).join(' ').toUpperCase();
            return `${prefix}Response: ${commandName} - Data: ${dataBytes}`;
        } else {
            return `${prefix}Response: ${commandName}`;
        }
    }

    // Check for other common EverTag patterns
    if (hexPairs.length >= 2) {
        // Hardware/Firmware info responses might start with different patterns
        if (firstByte >= 0x80 && firstByte <= 0xFF) {
            return `${prefix}System response 0x${firstByte.toString(16).toUpperCase()} - Data: ${hexPairs.slice(1).join(' ').toUpperCase()}`;
        }
    }

    // Fallback for any other payload on EverTag endpoints
    return `${prefix}Raw data - ${hexPairs.join(' ').toUpperCase()}`;
}

// Example invocation (as a comment):
// parseEverTagData('02 00 f5 05 a1 6c 0b 00 b8 04 02 86 0b 06 01 00', 238, 238)
// Output: Positioning: Seq: 2 | Anchor: 0x000B6CA1 RSSI: -59.0 dBm | Battery: 2950 mV | Mode: NRLS 

// At the end of the file, expose the parser and flag globally for use in ui_messages.js
window.parseEverTagData = parseEverTagData;
window.isTagParsingEnabled = isTagParsingEnabled; 