/**
 * ui_messages_enhanced.js
 *
 * Extends ui_messages.js to add source address range filtering and enhanced message filtering UI.
 * - Adds UI for min/max source address filter.
 * - Monkey-patches filter logic to support new filters.
 * - Uses global storage for custom filters and updates message list accordingly.
 * - Adds Clear Messages button functionality.
 * - See README.md for extension patterns.
 */

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize the enhanced messages UI
    setTimeout(initializeEnhancedMessagesUI, 500); // Delay to ensure uiMessages is fully loaded
    
    // Add Clear Messages button after a delay to ensure the UI is fully loaded
    setTimeout(addClearMessagesButton, 1000);
});

/**
 * Initialize the enhanced messages UI
 */
function initializeEnhancedMessagesUI() {
    if (!window.uiMessages) {
        console.error('uiMessages not found. Enhanced messages UI cannot be initialized.');
        return;
    }
    
    // Override the filter set function to handle source address range
    extendFilterSetFunction();
    
    // Add a monkey patch to the filter checking function
    addFilterCheckMonkeyPatch();
}

/**
 * Add a monkey patch to the filter checking function to handle source address range
 */
function addFilterCheckMonkeyPatch() {
    // We need to monkey patch the _513152 function in uiMessages
    // Since we can't access it directly, we'll use a different approach
    
    // Create a global function that will be called from our modified filter set function
    window.checkSourceAddressRange = function(message, filter) {
        if (!filter || !message) return true;
        
        // Check if this filter has a source address range
        if (filter.src_add_range_min !== undefined && filter.src_add_range_max !== undefined) {
            const srcAdd = parseInt(message.src_add);
            const minRange = parseInt(filter.src_add_range_min);
            const maxRange = parseInt(filter.src_add_range_max);
            
            // If the source address is outside the range, don't show the message
            if (srcAdd < minRange || srcAdd > maxRange) {
                return false;
            }
        }
        
        return true;
    };
}

/**
 * Extend the filter set function to handle source address range
 */
function extendFilterSetFunction() {
    // Store the original onFilterSet function
    const originalOnFilterSet = uiMessages.onFilterSet;
    
    // Override the onFilterSet function
    uiMessages.onFilterSet = function() {
        // Get the min and max values from the range inputs
        const minValue = document.getElementById('src_add_range_min_filter').value;
        const maxValue = document.getElementById('src_add_range_max_filter').value;
        
        // If both min and max are provided, create a special filter
        if (minValue !== '' && maxValue !== '') {
            // Create a new filter object
            const rangeFilter = {
                src_add_range_min: minValue,
                src_add_range_max: maxValue,
                enabled: true
            };
            
            // Add the filter to the active filters list
            const filtersList = document.getElementById('list_active_filters');
            const filterId = new Date().getTime(); // Use timestamp as a unique ID
            
            // Create a new list item for the filter
            const filterItem = document.createElement('li');
            filterItem.id = 'filter_id_' + filterId;
            
            // Create the filter display HTML
            filterItem.innerHTML = `
                <button id="enable_filter_button_${filterId}" class="fa fa-check-square-o button_filter" onclick="uiMessages.disableFilter(${filterId})"></button>&nbsp;&nbsp;
                <span padding_left=10px class="filter_label src_add_range">Src Add Range=${minValue}-${maxValue}</span>&nbsp;
                <button class="fa fa-close button_filter" onclick="uiMessages.removeFilter(${filterId})"></button>
            `;
            
            // Add the filter item to the list
            filtersList.appendChild(filterItem);
            
            // Store the filter in a global variable so we can access it later
            window.sourceAddressRangeFilters = window.sourceAddressRangeFilters || {};
            window.sourceAddressRangeFilters[filterId] = rangeFilter;
            
            // Add a monkey patch to the message filtering function
            monkeyPatchMessageFiltering();
            
            // Clear the range inputs
            document.getElementById('src_add_range_min_filter').value = '';
            document.getElementById('src_add_range_max_filter').value = '';
            
            // Update the message list
            uiMessages.onManualUpdate();
            
            return false; // Prevent the default form submission
        } else if (minValue !== '' || maxValue !== '') {
            // If only one of min or max is provided, show an error
            alert('Please provide both minimum and maximum values for the source address range filter.');
            return false; // Prevent the default form submission
        }
        
        // Otherwise, call the original function
        return originalOnFilterSet.call(uiMessages);
    };
    
    // Add the disableFilter and removeFilter functions for our custom filters
    const originalDisableFilter = uiMessages.disableFilter;
    uiMessages.disableFilter = function(filterId) {
        // Check if this is one of our custom filters
        if (window.sourceAddressRangeFilters && window.sourceAddressRangeFilters[filterId]) {
            const filter = window.sourceAddressRangeFilters[filterId];
            const button = document.getElementById('enable_filter_button_' + filterId);
            
            // Toggle the enabled state
            filter.enabled = !filter.enabled;
            
            // Update the button class
            if (filter.enabled) {
                button.className = 'fa fa-check-square-o button_filter';
            } else {
                button.className = 'fa fa-square-o button_filter';
            }
            
            // Update the message list
            uiMessages.onManualUpdate();
            return;
        }
        
        // Otherwise, call the original function
        return originalDisableFilter.call(uiMessages, filterId);
    };
    
    const originalRemoveFilter = uiMessages.removeFilter;
    uiMessages.removeFilter = function(filterId) {
        // Check if this is one of our custom filters
        if (window.sourceAddressRangeFilters && window.sourceAddressRangeFilters[filterId]) {
            // Remove the filter from our global variable
            delete window.sourceAddressRangeFilters[filterId];
            
            // Remove the filter item from the DOM
            const filterItem = document.getElementById('filter_id_' + filterId);
            if (filterItem) {
                filterItem.remove();
            }
            
            // Update the message list
            uiMessages.onManualUpdate();
            return;
        }
        
        // Otherwise, call the original function
        return originalRemoveFilter.call(uiMessages, filterId);
    };
}

/**
 * Monkey patch the message filtering function to handle our custom filters
 */
function monkeyPatchMessageFiltering() {
    // We need to monkey patch the _2ce1e9 function in uiMessages
    // Since we can't access it directly, we'll override the onManualUpdate function
    
    const originalOnManualUpdate = uiMessages.onManualUpdate;
    
    uiMessages.onManualUpdate = function() {
        // Call the original function first
        originalOnManualUpdate.call(uiMessages);
        
        // If we have no custom filters, we're done
        if (!window.sourceAddressRangeFilters || Object.keys(window.sourceAddressRangeFilters).length === 0) {
            return;
        }
        
        // Get all the messages currently displayed
        const messageList = document.querySelector('#message_list .list');
        if (!messageList) return;
        
        // Get all the message items
        const messageItems = messageList.querySelectorAll('li');
        
        // Check each message against our custom filters
        for (let i = 0; i < messageItems.length; i++) {
            const messageItem = messageItems[i];
            
            // Get the source address from the message item
            const srcAddElement = messageItem.querySelector('.src_add');
            if (!srcAddElement) continue;
            
            const srcAdd = parseInt(srcAddElement.textContent);
            if (isNaN(srcAdd)) continue;
            
            // Check the message against each of our custom filters
            let hideMessage = false;
            
            for (const filterId in window.sourceAddressRangeFilters) {
                const filter = window.sourceAddressRangeFilters[filterId];
                
                // Skip disabled filters
                if (!filter.enabled) continue;
                
                // Check if the source address is within the range
                const minRange = parseInt(filter.src_add_range_min);
                const maxRange = parseInt(filter.src_add_range_max);
                
                if (srcAdd < minRange || srcAdd > maxRange) {
                    hideMessage = true;
                    break;
                }
            }
            
            // Hide or show the message based on our filters
            if (hideMessage) {
                messageItem.style.display = 'none';
            }
        }
    };
}

/**
 * Add Clear Messages button next to Export CSV button
 */
function addClearMessagesButton() {
    try {
        // Strategy 1: Find the Update button directly
        let updateButton = null;
        let targetContainer = null;
        
        // Look for the Update button by text content
        const allButtons = document.querySelectorAll('button');
        for (let button of allButtons) {
            if (button.textContent && button.textContent.toLowerCase().includes('update')) {
                updateButton = button;
                targetContainer = button.parentElement;
                console.log('Found Update button:', updateButton);
                break;
            }
        }
        
        // Strategy 2: Look for the area with "Live update" checkbox as it's typically near Update button
        if (!targetContainer) {
            const liveUpdateElements = document.querySelectorAll('*');
            for (let element of liveUpdateElements) {
                if (element.textContent && element.textContent.includes('Live update')) {
                    targetContainer = element.parentElement;
                    console.log('Found Live update area as container');
                    break;
                }
            }
        }
        
        // Strategy 3: Look for Export CSV button and use its container
        if (!targetContainer) {
            const exportButton = document.querySelector('button[onclick*="exportCSV"], button[onclick*="export"], button[onclick*="Export"]');
            if (!exportButton) {
                for (let button of allButtons) {
                    if (button.textContent && button.textContent.toLowerCase().includes('export')) {
                        targetContainer = button.parentElement;
                        console.log('Found Export CSV button area');
                        break;
                    }
                }
            } else {
                targetContainer = exportButton.parentElement;
            }
        }
        
        if (targetContainer) {
            // Check if buttons already exist
            const existingClearButton = targetContainer.querySelector('button[title*="Clear all received messages"]');
            if (existingClearButton) {
                console.log('Clear Messages button already exists');
                return;
            }
            
            // Create a small inline container for our buttons
            const buttonContainer = document.createElement('span');
            buttonContainer.style.cssText = `
                display: inline-block;
                margin-left: 15px;
                vertical-align: middle;
            `;
            
            // Find and hide the original Export CSV button if it exists
            const originalExportButton = targetContainer.querySelector('button');
            if (originalExportButton && originalExportButton.textContent.toLowerCase().includes('export')) {
                originalExportButton.style.display = 'none';
            }
            
            // Create small Export CSV button
            const exportButton = document.createElement('button');
            exportButton.textContent = 'Export CSV';
            exportButton.title = 'Export messages to CSV file';
            exportButton.style.cssText = `
                background-color: #4CAF50;
                color: white;
                border: none;
                padding: 4px 8px;
                border-radius: 3px;
                cursor: pointer;
                font-size: 11px;
                font-weight: 500;
                height: 24px;
                min-width: 70px;
                margin-right: 5px;
                transition: background-color 0.2s;
            `;
            
            exportButton.onmouseover = function() {
                this.style.backgroundColor = '#45a049';
            };
            exportButton.onmouseout = function() {
                this.style.backgroundColor = '#4CAF50';
            };
            
            // Copy the onclick behavior from original Export CSV button if it exists
            if (originalExportButton && originalExportButton.onclick) {
                exportButton.onclick = originalExportButton.onclick;
            } else {
                // Fallback export functionality
                exportButton.onclick = function() {
                    if (typeof uiMessages !== 'undefined' && typeof uiMessages.exportCSV === 'function') {
                        uiMessages.exportCSV();
                    } else {
                        console.warn('Export CSV function not found');
                    }
                };
            }
            
            // Create small Clear Messages button
            const clearButton = document.createElement('button');
            clearButton.textContent = 'Clear';
            clearButton.title = 'Clear all received messages from the buffer';
            clearButton.style.cssText = `
                background-color: #f44336;
                color: white;
                border: none;
                padding: 4px 8px;
                border-radius: 3px;
                cursor: pointer;
                font-size: 11px;
                font-weight: 500;
                height: 24px;
                min-width: 50px;
                transition: background-color 0.2s;
            `;
            
            clearButton.onmouseover = function() {
                this.style.backgroundColor = '#d32f2f';
            };
            clearButton.onmouseout = function() {
                this.style.backgroundColor = '#f44336';
            };
            
            clearButton.onclick = function() {
                clearMessagesBuffer();
            };
            
            // Add buttons to container
            buttonContainer.appendChild(exportButton);
            buttonContainer.appendChild(clearButton);
            
            // Insert the button container at the end of the target container
            targetContainer.appendChild(buttonContainer);
            
            console.log('Small message control buttons added successfully');
        } else {
            console.warn('Could not find suitable container for message control buttons');
            
            // Fallback: create a floating mini toolbar
            const toolbarContainer = document.createElement('div');
            toolbarContainer.style.cssText = `
                position: fixed;
                top: 100px;
                right: 10px;
                z-index: 1000;
                display: flex;
                gap: 5px;
                background: rgba(255,255,255,0.95);
                padding: 5px;
                border-radius: 4px;
                box-shadow: 0 2px 8px rgba(0,0,0,0.15);
                border: 1px solid #ddd;
            `;
            
            // Small floating Export CSV button
            const exportButton = document.createElement('button');
            exportButton.textContent = 'CSV';
            exportButton.title = 'Export messages to CSV file';
            exportButton.style.cssText = `
                background-color: #4CAF50;
                color: white;
                border: none;
                padding: 4px 6px;
                border-radius: 3px;
                cursor: pointer;
                font-size: 10px;
                font-weight: 500;
                height: 22px;
                min-width: 35px;
            `;
            
            exportButton.onclick = function() {
                if (typeof uiMessages !== 'undefined' && typeof uiMessages.exportCSV === 'function') {
                    uiMessages.exportCSV();
                } else {
                    console.warn('Export CSV function not found');
                }
            };
            
            // Small floating Clear button
            const clearButton = document.createElement('button');
            clearButton.textContent = 'Clear';
            clearButton.title = 'Clear all messages';
            clearButton.style.cssText = `
                background-color: #f44336;
                color: white;
                border: none;
                padding: 4px 6px;
                border-radius: 3px;
                cursor: pointer;
                font-size: 10px;
                font-weight: 500;
                height: 22px;
                min-width: 40px;
            `;
            
            clearButton.onclick = function() {
                clearMessagesBuffer();
            };
            
            toolbarContainer.appendChild(exportButton);
            toolbarContainer.appendChild(clearButton);
            document.body.appendChild(toolbarContainer);
            
            console.log('Added floating mini toolbar for message controls');
        }
    } catch (error) {
        console.error('Error adding Clear Messages button:', error);
    }
}

/**
 * Clear the messages buffer and update the UI
 */
function clearMessagesBuffer() {
    try {
        let messagesCleared = 0;
        
        // Method 1: Clear mainApp messages buffer if available
        if (typeof mainApp !== 'undefined') {
            // Try different possible methods to clear messages
            if (typeof mainApp.clearMessages === 'function') {
                mainApp.clearMessages();
                console.log('Cleared messages using mainApp.clearMessages()');
            } else if (typeof mainApp.clearAllMessages === 'function') {
                mainApp.clearAllMessages();
                console.log('Cleared messages using mainApp.clearAllMessages()');
            } else if (mainApp.messages && Array.isArray(mainApp.messages)) {
                messagesCleared = mainApp.messages.length;
                mainApp.messages.length = 0; // Clear the array
                console.log(`Cleared ${messagesCleared} messages from mainApp.messages array`);
            } else if (typeof mainApp.getAllMessages === 'function') {
                const messages = mainApp.getAllMessages();
                if (Array.isArray(messages)) {
                    messagesCleared = messages.length;
                    messages.length = 0; // Clear the array
                    console.log(`Cleared ${messagesCleared} messages from getAllMessages array`);
                }
            }
        }
        
        // Method 2: Clear the visual message list in the UI
        const messageListElement = document.querySelector('#message_list .list, #message_list ul, #message_list');
        if (messageListElement) {
            const messageItems = messageListElement.querySelectorAll('li');
            const visibleMessages = messageItems.length;
            
            // Remove all message items
            messageItems.forEach(item => item.remove());
            console.log(`Cleared ${visibleMessages} messages from UI display`);
        }
        
        // Method 3: Clear any other message containers we can find
        const possibleMessageContainers = document.querySelectorAll('[id*="message"], [class*="message"]');
        possibleMessageContainers.forEach(container => {
            if (container.tagName === 'UL' || container.tagName === 'OL') {
                const items = container.querySelectorAll('li');
                if (items.length > 0) {
                    items.forEach(item => item.remove());
                    console.log(`Cleared ${items.length} items from container ${container.id || container.className}`);
                }
            }
        });
        
        // Method 4: Try to trigger any UI refresh functions
        if (typeof uiMessages !== 'undefined') {
            if (typeof uiMessages.onManualUpdate === 'function') {
                uiMessages.onManualUpdate();
                console.log('Triggered uiMessages.onManualUpdate()');
            }
            if (typeof uiMessages.refresh === 'function') {
                uiMessages.refresh();
                console.log('Triggered uiMessages.refresh()');
            }
        }
        
        // Show success message
        const totalCleared = messagesCleared || 'all';
        showClearStatus(`Successfully cleared ${totalCleared} messages from buffer`, 'success');
        
        // Update buffer counter immediately
        if (typeof messageFrequencyAnalyzer !== 'undefined' && 
            typeof messageFrequencyAnalyzer.updateBufferCounter === 'function') {
            setTimeout(() => messageFrequencyAnalyzer.updateBufferCounter(), 100);
        }
        
    } catch (error) {
        console.error('Error clearing messages buffer:', error);
        showClearStatus('Error clearing messages: ' + error.message, 'error');
    }
}

/**
 * Show status message for clear operation
 */
function showClearStatus(message, type) {
    try {
        // Try to use existing status system if available
        if (typeof uiEverTagConfig !== 'undefined' && typeof uiEverTagConfig.showStatus === 'function') {
            uiEverTagConfig.showStatus(message, type);
        } else {
            // Create a temporary status notification
            const statusDiv = document.createElement('div');
            statusDiv.className = `w3-panel w3-margin-top ${type === 'success' ? 'w3-pale-green w3-border-green' : 'w3-pale-red w3-border-red'}`;
            statusDiv.style.cssText = 'position: fixed; top: 60px; right: 20px; z-index: 1000; max-width: 300px;';
            statusDiv.innerHTML = `<span class="w3-closebtn" onclick="this.parentElement.style.display='none'">&times;</span><p>${message}</p>`;
            
            document.body.appendChild(statusDiv);
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                if (statusDiv.parentElement) {
                    statusDiv.remove();
                }
            }, 5000);
        }
    } catch (error) {
        // Fallback to console and alert
        console.log('Status:', message);
        if (type === 'error') {
            alert(message);
        }
    }
}

// Add a method to check if the enhanced messages UI is available
window.isEnhancedMessagesUIAvailable = function() {
    return true;
}; 