/**
 * ui_evertag_config.js
 *
 * Provides UI and logic for composing and sending EverTag configuration packets.
 * - Implements command selection, address input, and packet building for EverTag devices.
 * - Integrates with main message sending system and auto-populates fields from network data.
 * - Hooks into mainApp and uiMessages for enhanced config workflows.
 * - See README.md for extension patterns.
 */

var uiEverTagConfig = {
    // Configuration commands from the technical manual
    commands: [
        { id: 0x01, name: 'CBeacon advertisement interval', length: 1, description: 'Interval in seconds (1-10)', valueType: 'byte', min: 1, max: 10, unit: 's' },
        { id: 0x02, name: 'CBeacon TX power', length: 1, description: 'TX power in dBm (-128 to +127)', valueType: 'byte', min: -128, max: 127, unit: 'dBm' },
        { id: 0x03, name: 'Movement timeout', length: 1, description: 'Timeout in seconds = 2×value (10-500)', valueType: 'byte', min: 5, max: 250, unit: 's', multiplier: 2 },
        { id: 0x04, name: 'Movement threshold', length: 2, description: 'Threshold in mG (32-8000)', valueType: 'word', min: 32, max: 8000, unit: 'mG' },
        { id: 0x05, name: 'Movement positioning interval', length: 1, description: 'Positioning interval in seconds = 2×value (10-500)', valueType: 'byte', min: 5, max: 250, unit: 's', multiplier: 2 },
        { id: 0x06, name: 'Idle positioning interval', length: 2, description: 'Positioning interval in minutes (1-1440)', valueType: 'word', min: 1, max: 1440, unit: 'min' },
        { id: 0x07, name: 'Alert positioning interval', length: 1, description: 'Alert positioning interval in seconds = 2×value (10-500)', valueType: 'byte', min: 5, max: 250, unit: 's', multiplier: 2 },
        { id: 0x08, name: 'Alert timeout', length: 1, description: 'Timeout in seconds = 10×value (10-2500)', valueType: 'byte', min: 1, max: 250, unit: 's', multiplier: 10 },
        // { id: 0x09, name: 'LED intensity', length: 1, description: 'Intensity in percent (1-100)', valueType: 'byte', min: 1, max: 100, unit: '%' }, // OBSOLETED - LED intensity no longer supported
        { id: 0x0A, name: 'Start LED pattern', length: 4, description: 'LED pattern: repeat count + 6 blink codes (0=no blink, 1=short blink, 2=long blink, 500ms each)', valueType: 'ledpattern' },
        { id: 0x0B, name: 'Deactivate tag', length: 0, description: 'Deactivate the tag', valueType: 'none' },
        { id: 0x0C, name: 'Reset tag configuration', length: 0, description: 'Reset to default configuration', valueType: 'none' },
        { id: 0x0D, name: 'Enable CBeacon adverts', length: 0, description: 'Enable CBeacon advertisements', valueType: 'none' },
        { id: 0x0E, name: 'Disable CBeacon adverts', length: 0, description: 'Disable CBeacon advertisements', valueType: 'none' },
        { id: 0x0F, name: 'Get hardware info', length: 0, description: 'Request hardware information', valueType: 'none' },
        { id: 0x10, name: 'Get firmware info', length: 0, description: 'Request firmware information', valueType: 'none' },
        // { id: 0x11, name: 'Get sensor diagnostics', length: 0, description: 'Request sensor diagnostics', valueType: 'none' }, // OBSOLETED - Get sensor diagnostics no longer supported
        { id: 0x12, name: 'Get battery level', length: 0, description: 'Request battery level', valueType: 'none' },
        { id: 0x13, name: 'BloC report interval', length: 2, description: 'Interval in minutes (0-1440)', valueType: 'word', min: 0, max: 1440, unit: 'min' },
        { id: 0x14, name: 'Button shut-off timeout', length: 1, description: 'Timeout in seconds (0-120)', valueType: 'byte', min: 0, max: 120, unit: 's' },
        { id: 0x15, name: 'iBeacon mode', length: 1, description: '0=off, 1=on, 2=if no network', valueType: 'byte', min: 0, max: 2 },
        { id: 0x16, name: 'iBeacon UUID', length: 16, description: '16-byte UUID. i.e. F2746FD6-483D-11EE-BE56-0242AC120002', valueType: 'uuid' },
        { id: 0x17, name: 'iBeacon major/minor', length: 4, description: 'Major,Minor (e.g., 234,112). Default: Major=<0x00,nodeAddr[2]>, Minor=<nodeAddr[1],nodeAddr[0]>', valueType: 'ibeacon_major_minor' },
        { id: 0x18, name: 'NFC PIN', length: 8, description: '8-digit PIN', valueType: 'nfc' },
        { id: 0x19, name: 'Enable anchor tamper detection', length: 0, description: 'Enable tamper detection', valueType: 'none' },
        { id: 0x1A, name: 'Disable anchor tamper detection', length: 0, description: 'Disable tamper detection', valueType: 'none' },
        { id: 0x1B, name: 'EverTag-T temperature sample interval', length: 2, description: 'Interval in minutes (0-43200)', valueType: 'word', min: 0, max: 43200, unit: 'min' },
        { id: 0x1C, name: 'EverTag-SL sound/light sample interval', length: 2, description: 'Interval in minutes (1-43200)', valueType: 'word', min: 1, max: 43200, unit: 'min' },
	{ id: 0x1D, name: 'EverTag-IO Output Control', length: 2, description: 'Control output state', valueType: 'byte', min: 0, max: 255},
    ],

    // Current state
    selectedCommand: null,
    commandValue: '',
    targetType: 'broadcast-tags',
    addresses: [],
    sequenceNumber: 1,
    
    // App config specific state
    appConfigCommand: null,
    appConfigValue: '',
    appConfigTargetType: 'broadcast-tags', 
    appConfigAddresses: [],
    
    // Live update state
    liveUpdateEnabled: false,
    liveUpdateInterval: null,
    
    // Pending sink selection for auto-populating
    pendingSinkSelection: null,

    // Add to uiEverTagConfig object
    appConfigSequenceNumber: 1,
    tagConfigSequenceNumber: 1,

    // Add cached selection state
    cachedSelections: {
        networkId: null,
        gatewayId: null,
        sinkId: null
    },
    
    // Flag to track when we're restoring to avoid recursive saves
    isRestoringSelections: false,

    /**
     * Initialize the EverTag Configuration UI
     */
    initialize: function() {
        console.log('Initializing EverTag Configuration UI...');
        
        // Load sequence number from localStorage
        const savedSeq = localStorage.getItem('evertagConfigSequence');
        if (savedSeq) {
            this.sequenceNumber = parseInt(savedSeq, 10);
        }
        
        // Load app config sequence number from localStorage
        const savedAppConfigSeq = localStorage.getItem('appConfigSequence');
        if (savedAppConfigSeq) {
            this.appConfigSequenceNumber = parseInt(savedAppConfigSeq, 10);
        }
        
        // Load tag config sequence number from localStorage
        const savedTagSeq = localStorage.getItem('tagConfigSequence');
        if (savedTagSeq) {
            this.tagConfigSequenceNumber = parseInt(savedTagSeq, 10);
        }
        
        // Load cached selections from localStorage
        const savedCachedSelections = localStorage.getItem('evertagCachedSelections');
        if (savedCachedSelections) {
            try {
                this.cachedSelections = JSON.parse(savedCachedSelections);
                console.log('Loaded cached selections:', this.cachedSelections);
            } catch (e) {
                console.error('Error parsing cached selections:', e);
            }
        }
        
        // Only initialize full UI elements if they exist (for the standalone config page)
        const sequenceNumberElement = document.getElementById('sequence_number');
        if (sequenceNumberElement) {
            sequenceNumberElement.textContent = this.sequenceNumber;
            
            // Populate commands dropdown (only for full config page)
            this.populateCommandsDropdown();
            
            // Set initial target type (only for full config page)
            this.onTargetTypeChanged('broadcast-tags');
        }
        
        // Initialize app config UI elements if they exist (for messages page)
        this.loadAppConfigCommands();
        
        // Always try to populate networks dropdown (for both standalone and messages page)
        this.populateNetworksDropdown();
        
        // Start live update if checkbox exists and is checked
        const liveUpdateCheckbox = document.getElementById('live_update_evertag');
        if (liveUpdateCheckbox && liveUpdateCheckbox.checked) {
            this.onLiveUpdateChange(liveUpdateCheckbox);
        }
        
        // Hook into mainApp's gateway config received event (like gateway page does)
        if (typeof mainApp !== 'undefined') {
            // Store original callback if it exists
            this.originalOnGatewayConfigReceived = mainApp.onGatewayConfigReceived;
            
            // Override with our enhanced callback
            mainApp.onGatewayConfigReceived = (gatewayId, configs) => {
                // Call original callback first
                if (this.originalOnGatewayConfigReceived) {
                    this.originalOnGatewayConfigReceived(gatewayId, configs);
                }
                
                // Then call our callback for auto-population
                this.onGatewayConfigReceived(gatewayId);
            };
        }
        
        // Update the input field
        const tagSeqInput = document.getElementById('tag_config_sequence');
        if (tagSeqInput) {
            tagSeqInput.value = this.tagConfigSequenceNumber;
            tagSeqInput.onchange = () => {
                this.tagConfigSequenceNumber = parseInt(tagSeqInput.value, 10);
                this.updatePacket();
            };
        }
        
        // Update the app config tag sequence input field
        const appConfigTagSeqInput = document.getElementById('appconfig_tag_config_sequence');
        if (appConfigTagSeqInput) {
            appConfigTagSeqInput.value = this.tagConfigSequenceNumber;
            appConfigTagSeqInput.onchange = () => {
                this.tagConfigSequenceNumber = parseInt(appConfigTagSeqInput.value, 10);
                localStorage.setItem('tagConfigSequence', this.tagConfigSequenceNumber.toString());
                this.updateAppConfigPacketDisplay();
            };
        }
        
        // Restore cached selections after networks are populated
        this.restoreCachedSelections();
        
        console.log('EverTag Configuration UI initialized');
    },

    /**
     * Initialize the EverTag Configuration UI for Messages page
     */
    initializeForMessages: function() {
        console.log('Initializing EverTag Configuration UI for Messages page...');
        
        // Load sequence numbers from localStorage
        const savedSeq = localStorage.getItem('evertagConfigSequence');
        if (savedSeq) {
            this.sequenceNumber = parseInt(savedSeq, 10);
        }
        
        const savedAppConfigSeq = localStorage.getItem('appConfigSequence');
        if (savedAppConfigSeq) {
            this.appConfigSequenceNumber = parseInt(savedAppConfigSeq, 10);
        }
        
        const savedTagSeq = localStorage.getItem('tagConfigSequence');
        if (savedTagSeq) {
            this.tagConfigSequenceNumber = parseInt(savedTagSeq, 10);
        }
        
        // Initialize app config UI elements
        this.loadAppConfigCommands();
        
        // Always populate networks dropdown on initial load (no live update dependency)
        this.populateNetworksDropdown();
        
        // Hide the live update checkbox for app config (since we don't need it)
        setTimeout(() => {
            const liveUpdateCheckbox = document.getElementById('live_update_evertag');
            if (liveUpdateCheckbox) {
                const liveUpdateContainer = liveUpdateCheckbox.closest('label') || liveUpdateCheckbox.parentElement;
                if (liveUpdateContainer) {
                    liveUpdateContainer.style.display = 'none';
                    console.log('Debug: Hidden live update checkbox for app config');
                }
            }
        }, 500);
        
        // Update the tag sequence input field
        const tagSeqInput = document.getElementById('tag_config_sequence');
        if (tagSeqInput) {
            tagSeqInput.value = this.tagConfigSequenceNumber;
            tagSeqInput.onchange = () => {
                this.tagConfigSequenceNumber = parseInt(tagSeqInput.value, 10);
                localStorage.setItem('tagConfigSequence', this.tagConfigSequenceNumber.toString());
                this.updateAppConfigPacketDisplay();
            };
        }
        
        // Update the app config tag sequence input field
        const appConfigTagSeqInput = document.getElementById('appconfig_tag_config_sequence');
        if (appConfigTagSeqInput) {
            appConfigTagSeqInput.value = this.tagConfigSequenceNumber;
        }
        
        // Hook into mainApp's gateway config received event
        if (typeof mainApp !== 'undefined') {
            this.originalOnGatewayConfigReceived = mainApp.onGatewayConfigReceived;
            mainApp.onGatewayConfigReceived = (gatewayId, configs) => {
                if (this.originalOnGatewayConfigReceived) {
                    this.originalOnGatewayConfigReceived(gatewayId, configs);
                }
                this.onGatewayConfigReceived(gatewayId);
            };
        }
        
        // Restore cached selections after initialization
        this.restoreCachedSelections();
        
        console.log('EverTag Configuration UI for Messages page initialized');
    },

    /**
     * Initialize the EverTag Configuration UI for Messages page (without live update dependency)
     */
    initializeAppConfigMode: function() {
        console.log('Initializing EverTag App Config mode...');
        
        // Load sequence numbers from localStorage
        const savedTagSeq = localStorage.getItem('tagConfigSequence');
        if (savedTagSeq) {
            this.tagConfigSequenceNumber = parseInt(savedTagSeq, 10);
        }
        
        // Initialize app config UI elements
        this.loadAppConfigCommands();
        
        // Always populate networks dropdown on initial load (no live update checkbox needed)
        this.populateNetworksDropdown();
        
        // Hook into mainApp's gateway config received event
        if (typeof mainApp !== 'undefined') {
            this.originalOnGatewayConfigReceived = mainApp.onGatewayConfigReceived;
            mainApp.onGatewayConfigReceived = (gatewayId, configs) => {
                if (this.originalOnGatewayConfigReceived) {
                    this.originalOnGatewayConfigReceived(gatewayId, configs);
                }
                this.onGatewayConfigReceived(gatewayId);
            };
        }
        
        console.log('EverTag App Config mode initialized');
    },

    /**
     * Add observer to populate networks when the App config wizard section becomes visible
     */
    addVisibilityObserver: function() {
        // Find the App config wizard details element
        const quickDiagnosticsDetails = document.querySelector('details summary b');
        if (quickDiagnosticsDetails && quickDiagnosticsDetails.textContent.includes('App config wizard')) {
            const detailsElement = quickDiagnosticsDetails.closest('details');
            
            if (detailsElement) {
                // Listen for when the details element is opened
                detailsElement.addEventListener('toggle', () => {
                    if (detailsElement.open) {
                        console.log('App config wizard section opened, populating networks...');
                        setTimeout(() => {
                            this.populateNetworksDropdown();
                        }, 100);
                    }
                });
            }
        }
        
        // Also use a MutationObserver to watch for when the messages page becomes visible
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.type === 'attributes' && mutation.attributeName === 'style') {
                    const messageMode = document.getElementById('message_mode');
                    if (messageMode && messageMode.style.display !== 'none') {
                        console.log('Messages page became visible, ensuring networks are populated...');
                        setTimeout(() => {
                            this.populateNetworksDropdown();
                        }, 500);
                    }
                }
            });
        });
        
        const messageMode = document.getElementById('message_mode');
        if (messageMode) {
            observer.observe(messageMode, {
                attributes: true,
                attributeFilter: ['style']
            });
        }
    },

    /**
     * Handle live update change for networks dropdown (matches pattern from other components)
     */
    onLiveUpdateChange: function(checkbox) {
        this.liveUpdateEnabled = checkbox.checked;
        
        if (this.liveUpdateEnabled) {
            console.log('EverTag live update enabled');
            // Immediately refresh the networks
            this.populateNetworksDropdown();
            
            // Start auto-update interval (every 3 seconds when live update is enabled)
            this.liveUpdateInterval = setInterval(() => {
                if (this.liveUpdateEnabled) { // Double check in case something changed
                    this.populateNetworksDropdown();
                }
            }, 3000);
        } else {
            console.log('EverTag live update disabled');
            // Stop the auto-update interval
            if (this.liveUpdateInterval) {
                clearInterval(this.liveUpdateInterval);
                this.liveUpdateInterval = null;
            }
        }
    },

    /**
     * Manual update function (called by the Update button and potentially by live update system)
     */
    onManualUpdate: function() {
        console.log('Manual update triggered for EverTag networks');
        this.forcePopulateNetworks();
    },

    /**
     * Populate the commands dropdown
     */
    populateCommandsDropdown: function() {
        const select = document.getElementById('evertag_command_select');
        select.innerHTML = '<option value="">-- Select Command --</option>';
        
        this.commands.forEach(cmd => {
            const option = document.createElement('option');
            option.value = cmd.id;
            option.textContent = `0x${cmd.id.toString(16).toUpperCase().padStart(2, '0')} - ${cmd.name}`;
            select.appendChild(option);
        });
    },

    /**
     * Populate the networks dropdown with available networks
     */
    populateNetworksDropdown: function() {
        const select = document.getElementById('evertag_network_id');
        if (!select) {
            return;
        }
        
        // Save the currently selected value before clearing
        const currentSelection = select.value;
        
        // Clear existing options except the default
        select.innerHTML = '<option value="">-- Select Network --</option>';
        
        try {
            // Get all unique network IDs from messages and gateways
            const networks = new Set();
            
            // Get networks from messages if available
            if (typeof mainApp !== 'undefined' && mainApp.getAllMessages) {
                const messages = mainApp.getAllMessages();
                
                if (Array.isArray(messages)) {
                    messages.forEach((message) => {
                        if (message.network_id && message.network_id !== '-' && message.network_id !== 'undefined') {
                            networks.add(message.network_id);
                        }
                    });
                }
            }
            
            // Get networks from gateway configurations if available
            if (typeof mainApp !== 'undefined' && mainApp.getAllGateways) {
                const gateways = mainApp.getAllGateways();
                
                if (gateways && typeof gateways === 'object') {
                    Object.values(gateways).forEach(gateway => {
                        if (gateway.configs && Array.isArray(gateway.configs)) {
                            gateway.configs.forEach(config => {
                                if (config.network_address && config.network_address !== 0) {
                                    networks.add(config.network_address);
                                }
                            });
                        }
                    });
                }
            }
            
            // Convert set to sorted array and populate dropdown
            const sortedNetworks = Array.from(networks).sort((a, b) => {
                // Try numeric sort first
                const numA = parseInt(a, 10);
                const numB = parseInt(b, 10);
                if (!isNaN(numA) && !isNaN(numB)) {
                    return numA - numB;
                }
                // Fall back to string sort
                return String(a).localeCompare(String(b));
            });
            
            let selectionRestored = false;
            sortedNetworks.forEach(networkId => {
                const option = document.createElement('option');
                option.value = networkId;
                option.textContent = `Network ${networkId}`;
                select.appendChild(option);
                
                // Restore the previous selection if it still exists
                if (currentSelection && networkId == currentSelection) {
                    option.selected = true;
                    selectionRestored = true;
                }
            });
            
            // If the previous selection no longer exists, clear the selection
            if (currentSelection && !selectionRestored) {
                select.value = '';
            }
            
        } catch (error) {
            console.error('Error populating networks dropdown:', error);
        }
    },

    /**
     * Restore cached network/gateway/sink selections
     */
    restoreCachedSelections: function() {
        console.log('Attempting to restore cached selections:', this.cachedSelections);
        
        // Only restore if we have valid cached selections
        if (!this.cachedSelections.networkId) {
            console.log('No cached network selection to restore');
            return;
        }
        
        // Set flag to prevent saving during restoration
        this.isRestoringSelections = true;
        
        // Delay the restoration to ensure dropdowns are populated
        setTimeout(() => {
            const networkSelect = document.getElementById('evertag_network_id');
            if (networkSelect) {
                // Check if the cached network still exists in the dropdown
                const networkOption = Array.from(networkSelect.options).find(option => option.value === this.cachedSelections.networkId);
                if (networkOption) {
                    console.log('Restoring network selection:', this.cachedSelections.networkId);
                    networkSelect.value = this.cachedSelections.networkId;
                    
                    // Trigger the onchange event to populate gateways
                    this.onNetworkChanged(this.cachedSelections.networkId);
                    
                    // Restore gateway selection if available
                    if (this.cachedSelections.gatewayId) {
                        setTimeout(() => {
                            const gatewaySelect = document.getElementById('evertag_gateway_id');
                            if (gatewaySelect) {
                                const gatewayOption = Array.from(gatewaySelect.options).find(option => option.value === this.cachedSelections.gatewayId);
                                if (gatewayOption) {
                                    console.log('Restoring gateway selection:', this.cachedSelections.gatewayId);
                                    gatewaySelect.value = this.cachedSelections.gatewayId;
                                    
                                    // Trigger the onchange event to populate sinks
                                    this.onGatewayChanged(this.cachedSelections.gatewayId);
                                    
                                    // Restore sink selection if available
                                    if (this.cachedSelections.sinkId) {
                                        setTimeout(() => {
                                            const sinkSelect = document.getElementById('evertag_sink_id');
                                            if (sinkSelect) {
                                                const sinkOption = Array.from(sinkSelect.options).find(option => option.value === this.cachedSelections.sinkId);
                                                if (sinkOption) {
                                                    console.log('Restoring sink selection:', this.cachedSelections.sinkId);
                                                    sinkSelect.value = this.cachedSelections.sinkId;
                                                    
                                                    // Trigger the onchange event for sink
                                                    this.onSinkChanged(this.cachedSelections.sinkId);
                                                } else {
                                                    console.log('Cached sink no longer available:', this.cachedSelections.sinkId);
                                                    this.cachedSelections.sinkId = null;
                                                    localStorage.setItem('evertagCachedSelections', JSON.stringify(this.cachedSelections));
                                                }
                                            }
                                        }, 500); // Delay for sink restoration
                                    }
                                } else {
                                    console.log('Cached gateway no longer available:', this.cachedSelections.gatewayId);
                                    this.cachedSelections.gatewayId = null;
                                    this.cachedSelections.sinkId = null;
                                    localStorage.setItem('evertagCachedSelections', JSON.stringify(this.cachedSelections));
                                }
                            }
                        }, 500); // Delay for gateway restoration
                    }
                } else {
                    console.log('Cached network no longer available:', this.cachedSelections.networkId);
                    // Clear all cached selections if network is no longer available
                                         this.cachedSelections = { networkId: null, gatewayId: null, sinkId: null };
                     localStorage.setItem('evertagCachedSelections', JSON.stringify(this.cachedSelections));
                 }
             }
             
             // Clear the restoration flag after all operations
             setTimeout(() => {
                 this.isRestoringSelections = false;
                 console.log('Restoration complete, flag cleared');
             }, 2000);
         }, 1000); // Initial delay to ensure networks are populated
     },

    /**
     * Handle command selection change
     */
    onCommandChanged: function(commandId) {
        if (!commandId) {
            this.selectedCommand = null;
            document.getElementById('command_description').classList.add('w3-hide');
            document.getElementById('command_value_section').classList.add('w3-hide');
            this.updateSendButton();
            return;
        }

        this.selectedCommand = this.commands.find(cmd => cmd.id == commandId);
        
        if (this.selectedCommand) {
            // Show description
            document.getElementById('command_description_text').textContent = this.selectedCommand.description;
            document.getElementById('command_description').classList.remove('w3-hide');
            
            // Show/hide value input based on command type
            if (this.selectedCommand.valueType === 'none') {
                document.getElementById('command_value_section').classList.add('w3-hide');
                this.commandValue = '';
            } else {
                this.setupValueInput();
                document.getElementById('command_value_section').classList.remove('w3-hide');
            }
        }
        
        this.updatePacket();
        this.updateSendButton();
    },

    /**
     * Setup the value input field based on command type
     */
    setupValueInput: function() {
        const input = document.getElementById('command_value_input');
        const help = document.getElementById('command_value_help');
        const cmd = this.selectedCommand;
        
        input.value = '';
        this.commandValue = '';
        
        // Clear all attributes first to prevent carryover from previous commands
        input.removeAttribute('min');
        input.removeAttribute('max');
        input.removeAttribute('maxLength');
        input.removeAttribute('pattern');
        input.removeAttribute('title');
        input.placeholder = '';
        
        switch (cmd.valueType) {
            case 'byte':
            case 'word':
            case 'dword':
                input.type = 'number';
                const actualMin = cmd.multiplier ? (cmd.min || 0) * cmd.multiplier : cmd.min;
                const actualMax = cmd.multiplier ? (cmd.max || 255) * cmd.multiplier : cmd.max;
                input.min = actualMin;
                input.max = actualMax;
                input.title = `Range: ${actualMin || 'N/A'} - ${actualMax || 'N/A'} ${cmd.unit || ''}`;
                help.textContent = `Range: ${actualMin || 'N/A'} - ${actualMax || 'N/A'} ${cmd.unit || ''}`;
                break;
                
            case 'uuid':
                input.type = 'text';
                input.placeholder = 'Enter UUID or copy default below';
                input.maxLength = 36; // Standard UUID format length
                input.pattern = '[0-9A-Fa-f]{8}-[0-9A-Fa-f]{4}-[0-9A-Fa-f]{4}-[0-9A-Fa-f]{4}-[0-9A-Fa-f]{12}'; // UUID pattern
                input.title = 'Enter a valid UUID in format: XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX';
                help.innerHTML = `
                    16-byte UUID. i.e. F2746FD6-483D-11EE-BE56-0242AC120002
                `;
                break;
                
            case 'nfc':
                input.type = 'text';
                input.maxLength = 8;
                input.placeholder = '12345678';
                input.pattern = '[0-9]{1,8}'; // Only digits, 1-8 characters
                input.title = '8-digit PIN (e.g., 12345678) - each digit becomes 0x01-0x09, sent in reverse order';
                input.setAttribute('data-original-title', input.title);
                help.innerHTML = `
                    <strong>NFC PIN Format:</strong> 8-digit PIN code [0-9]<br>
                    <strong>Example:</strong> 12345678 → [0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08]<br>
                    <strong>Byte Order:</strong> Little-endian (reversed) → [0x08,0x07,0x06,0x05,0x04,0x03,0x02,0x01]<br>
                    <strong>Each digit:</strong> 1→0x01, 2→0x02, ... 9→0x09, 0→0x00
                `;
                break;
                
            case 'ledpattern':
                input.type = 'text';
                input.placeholder = '5,21,12,21';
                input.title = 'LED Pattern: repeat_count,byte1,byte2,byte3 (digits 0,1,2 only)';
                help.innerHTML = `
                    <strong>LED Pattern Format:</strong> repeat_count,byte1,byte2,byte3<br>
                    <strong>Repeat Count:</strong> Number of times to repeat (total duration = 4×count seconds)<br>
                    <strong>Pattern Bytes:</strong> Each byte contains exactly 2 blink codes (6 codes total):<br>
                    &nbsp;&nbsp;• 0 = no blink (500ms), 1 = short blink (500ms), 2 = long blink (500ms)<br>
                    <strong>Example:</strong> 5,21,12,21 = 5 repeats of pattern: 2-1-1-2-2-1 (3 seconds + 1 second silence)<br>
                    <strong>Structure:</strong> 3 seconds of pattern (6×500ms) + 1 second automatic silence<br>
                    <strong>Reference:</strong> 0x05211221 = repeat 5 times, pattern bytes 21,12,21
                `;
                break;
                
            case 'pattern':
                input.type = 'text';
                input.placeholder = '5,34,34,34';
                input.title = 'Comma-separated: repeat count, pattern byte 1, pattern byte 2, pattern byte 3';
                help.textContent = 'Comma-separated: repeat count, pattern byte 1, pattern byte 2, pattern byte 3';
                break;
                
            case 'ibeacon_major_minor':
                input.type = 'text';
                input.placeholder = '234,112';
                input.pattern = '[0-9]+,[0-9]+'; // Require digits,digits pattern
                input.title = 'Enter Major,Minor as decimal values (e.g., 234,112)';
                input.setAttribute('data-original-title', input.title);
                help.innerHTML = `
                    <strong>iBeacon Major/Minor Format:</strong> Major,Minor (decimal values)<br>
                    <strong>Example:</strong> 234,0 → Major=0x00EA, Minor=0x0000 → [EA 00 00 00]<br>
                    <strong>Byte Order:</strong> Little-endian (Major LSB, Major MSB, Minor LSB, Minor MSB)<br>
                    <strong>Default:</strong> Major=&lt;0x00,nodeAddr[2]&gt;, Minor=&lt;nodeAddr[1],nodeAddr[0]&gt;<br>
                    <strong>Range:</strong> Major/Minor 0-65535 each
                `;
                break;
                
            default:
                input.type = 'text';
                input.title = cmd.description || 'Enter value for this command';
                help.textContent = cmd.description || '';
        }
    },

    /**
     * Validate command value based on command type
     * @param {object} command - Command object
     * @param {string} value - User input value
     * @returns {object} - {isValid: boolean, error: string}
     */
    validateCommandValue: function(command, value) {
        if (!command || !command.valueType || command.valueType === 'none') {
            return { isValid: true, error: '' };
        }
        
        const val = value.trim();
        if (!val) {
            return { isValid: false, error: 'Value is required' };
        }
        
        switch (command.valueType) {
            case 'ibeacon_major_minor': {
                // Must contain exactly one comma
                if (!val.includes(',')) {
                    return { isValid: false, error: 'Must contain comma-separated Major,Minor values (e.g., 234,112)' };
                }
                
                const parts = val.split(',');
                if (parts.length !== 2) {
                    return { isValid: false, error: 'Must contain exactly two values separated by comma (Major,Minor)' };
                }
                
                // Check each part is a valid number
                const major = parseInt(parts[0].trim(), 10);
                const minor = parseInt(parts[1].trim(), 10);
                
                if (isNaN(major) || isNaN(minor)) {
                    return { isValid: false, error: 'Both Major and Minor must be valid numbers' };
                }
                
                if (major < 0 || major > 65535) {
                    return { isValid: false, error: 'Major must be between 0 and 65535' };
                }
                
                if (minor < 0 || minor > 65535) {
                    return { isValid: false, error: 'Minor must be between 0 and 65535' };
                }
                
                return { isValid: true, error: '' };
            }
            
            case 'nfc': {
                // Remove non-digits to check actual content
                const digits = val.replace(/[^0-9]/g, '');
                
                if (val !== digits) {
                    return { isValid: false, error: 'NFC PIN must contain only digits (0-9)' };
                }
                
                if (digits.length === 0) {
                    return { isValid: false, error: 'NFC PIN cannot be empty' };
                }
                
                if (digits.length > 8) {
                    return { isValid: false, error: 'NFC PIN cannot be longer than 8 digits' };
                }
                
                return { isValid: true, error: '' };
            }
            
            case 'uuid': {
                const uuid = val.replace(/-/g, '');
                if (!/^[0-9a-fA-F]{32}$/.test(uuid)) {
                    return { isValid: false, error: 'Invalid UUID format. Must be 32 hex characters with dashes (XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX)' };
                }
                return { isValid: true, error: '' };
            }
            
            case 'ledpattern': {
                const parts = val.split(',');
                if (parts.length < 2 || parts.length > 4) {
                    return { isValid: false, error: 'LED pattern must have 2-4 comma-separated values' };
                }
                
                // Check repeat count
                const repeatCount = parseInt(parts[0].trim(), 10);
                if (isNaN(repeatCount) || repeatCount < 0 || repeatCount > 255) {
                    return { isValid: false, error: 'Repeat count must be a number between 0 and 255' };
                }
                
                // Check pattern bytes (only digits 0, 1, 2 allowed)
                for (let i = 1; i < parts.length; i++) {
                    const pattern = parts[i].trim();
                    if (!/^[0-2]{1,2}$/.test(pattern)) {
                        return { isValid: false, error: 'Pattern bytes must contain only digits 0, 1, or 2 (up to 2 digits each)' };
                    }
                }
                
                return { isValid: true, error: '' };
            }
            
            case 'byte':
            case 'word':
            case 'dword': {
                const num = parseInt(val, 10);
                if (isNaN(num)) {
                    return { isValid: false, error: 'Must be a valid number' };
                }
                
                if (command.min !== undefined || command.max !== undefined) {
                    const actualMin = command.multiplier ? (command.min || 0) * command.multiplier : (command.min || -Infinity);
                    const actualMax = command.multiplier ? (command.max || 255) * command.multiplier : (command.max || Infinity);
                    
                    if (num < actualMin || num > actualMax) {
                        return { isValid: false, error: `Value must be between ${actualMin} and ${actualMax}` };
                    }
                }
                
                return { isValid: true, error: '' };
            }
            
            default:
                return { isValid: true, error: '' };
        }
    },

    /**
     * Handle value input change
     */
    onValueChanged: function(value) {
        this.commandValue = value;
        
        // Validate the input
        const validation = this.validateCommandValue(this.selectedCommand, value);
        const input = document.getElementById('command_value_input');
        const helpElement = document.getElementById('command_value_help');
        
        // Show validation error if invalid
        if (!validation.isValid && value.trim()) {
            input.style.borderColor = '#f44336';
            input.title = validation.error;
            // Temporarily show error in help text
            const originalHelp = helpElement.innerHTML;
            helpElement.innerHTML = `<span style="color: #f44336;"><strong>Error:</strong> ${validation.error}</span>`;
            setTimeout(() => {
                helpElement.innerHTML = originalHelp;
                input.style.borderColor = '';
                input.title = input.getAttribute('data-original-title') || '';
            }, 3000);
        } else {
            input.style.borderColor = '';
        }
        
        this.updatePacket();
        this.updateSendButton();
    },

    /**
     * Handle target type change
     */
    onTargetTypeChanged: function(targetType) {
        this.targetType = targetType;
        const addressSection = document.getElementById('address_section');
        const addBtn = document.getElementById('add_address_btn');
        
        if (targetType === 'unicast' || targetType === 'multicast') {
            addressSection.classList.remove('w3-hide');
            
            if (targetType === 'unicast') {
                this.addresses = [''];
                addBtn.classList.add('w3-hide');
            } else {
                this.addresses = ['', ''];
                addBtn.classList.remove('w3-hide');
            }
            
            this.updateAddressInputs();
        } else {
            addressSection.classList.add('w3-hide');
            this.addresses = [];
        }
        
        this.updatePacket();
        this.updateSendButton();
    },

    /**
     * Update address input fields
     */
    updateAddressInputs: function() {
        const container = document.getElementById('address_inputs');
        container.innerHTML = '';
        
        this.addresses.forEach((addr, index) => {
            const input = document.createElement('input');
            input.className = 'w3-input w3-border w3-margin-bottom';
            input.type = 'text';
            input.placeholder = '00381D77';
            input.value = addr;
            input.onchange = () => this.onAddressChanged();
            
            if (this.targetType === 'multicast' && this.addresses.length > 1) {
                const wrapper = document.createElement('div');
                wrapper.style.display = 'flex';
                wrapper.style.alignItems = 'center';
                wrapper.style.marginBottom = '8px';
                
                wrapper.appendChild(input);
                
                const removeBtn = document.createElement('button');
                removeBtn.className = 'w3-button w3-small w3-red';
                removeBtn.innerHTML = '<i class="fa fa-minus"></i>';
                removeBtn.style.marginLeft = '8px';
                removeBtn.onclick = () => this.removeAddress(index);
                
                wrapper.appendChild(removeBtn);
                container.appendChild(wrapper);
            } else {
                container.appendChild(input);
            }
        });
    },

    /**
     * Handle address input change
     */
    onAddressChanged: function() {
        const inputs = document.querySelectorAll('#address_inputs input');
        this.addresses = Array.from(inputs).map(input => input.value.trim());
        this.updatePacket();
        this.updateSendButton();
    },

    /**
     * Add a new address input (multicast only)
     */
    addAddress: function() {
        if (this.addresses.length < 15) {
            this.addresses.push('');
            this.updateAddressInputs();
        }
    },

    /**
     * Remove an address input
     */
    removeAddress: function(index) {
        if (this.addresses.length > 1) {
            this.addresses.splice(index, 1);
            this.updateAddressInputs();
            this.updatePacket();
            this.updateSendButton();
        }
    },

    /**
     * Parse a hex address string to 4 bytes (little-endian)
     */
    parseAddress: function(address) {
        const cleaned = address.replace(/[^0-9a-fA-F]/g, '');
        const hex = cleaned.padStart(8, '0');
        
        // Convert to little-endian bytes
        const bytes = [];
        for (let i = 6; i >= 0; i -= 2) {
            bytes.push(parseInt(hex.substr(i, 2), 16));
        }
        
        return bytes;
    },

    /**
     * Convert a decimal or hex address to 4 bytes (little-endian)
     * Handles both decimal numbers (like 2131901) and hex strings (like "00381D7D")
     */
    parseAddressDecimalOrHex: function(address) {
        let addressValue;
        
        // Check if it's a hex string (contains letters or starts with 0x)
        if (/[a-fA-F]/.test(address) || address.toLowerCase().startsWith('0x')) {
            // It's hex - use existing parseAddress method
            console.log(`[AddressParse] Treating "${address}" as hex`);
            return this.parseAddress(address);
        } else {
            // It's decimal - convert to 32-bit integer first
            addressValue = parseInt(address, 10);
            if (isNaN(addressValue)) {
                console.warn('Invalid address:', address);
                addressValue = 0;
            }
            console.log(`[AddressParse] Treating "${address}" as decimal: ${addressValue} (0x${addressValue.toString(16).toUpperCase()})`);
        }
        
        // Convert decimal to little-endian bytes
        const bytes = [];
        bytes.push(addressValue & 0xFF);           // Byte 0 (LSB)
        bytes.push((addressValue >> 8) & 0xFF);   // Byte 1
        bytes.push((addressValue >> 16) & 0xFF);  // Byte 2
        bytes.push((addressValue >> 24) & 0xFF);  // Byte 3 (MSB)
        
        console.log(`[AddressParse] Little-endian bytes: [${bytes.map(b => '0x' + b.toString(16).padStart(2, '0').toUpperCase()).join(', ')}]`);
        return bytes;
    },

    /**
     * Unified function to build EverTag config/app config packet as hex string
     * @param {object} command - Command object (from commands array)
     * @param {string} value - Command value (user input)
     * @param {string} targetType - Target type (e.g., 'broadcast-tags', 'unicast')
     * @param {Array<string>} addresses - Array of address strings
     * @param {number} sequenceNumber - Sequence number (integer)
     * @returns {string} - Hex string (e.g., 'B0 07 02 80 12')
     */
    buildEverTagPacket: function(command, value, targetType, addresses, sequenceNumber) {
        console.log('[UnifiedPacket] Building packet with:', {command, value, targetType, addresses, sequenceNumber});
        const packet = [];
        // Header: CONFIG_ID (0xB0)
        packet.push(0xB0);
        // SEQ: Sequence number (ensure number)
        const seqNum = typeof sequenceNumber === 'string' ? parseInt(sequenceNumber.trim(), 10) : sequenceNumber;
        packet.push(seqNum & 0xFF);
        // Calculate MSG_TYPE byte and handle addresses
        let msgType = 0;
        let addressBytes = [];
        const tgtType = typeof targetType === 'string' ? targetType.trim() : targetType;
        switch (tgtType) {
            case 'broadcast-tags':
                msgType = 0b10000000; // T=1
                break;
            case 'broadcast-anchors':
                msgType = 0b01000000; // A=1
                break;
            case 'unicast':
                msgType = 0b00000000; // LLLL=0 (unicast = 1 address, NA=0)
                if (addresses && addresses.length > 0 && addresses[0] && addresses[0].trim()) {
                    addressBytes = this.parseAddressDecimalOrHex(addresses[0].trim());
                }
                break;
            case 'multicast':
                const validAddresses = (addresses || []).filter(addr => addr && addr.trim());
                // MSG_TYPE LLLL field = Number of Additional Addresses (NA = total - 1)
                // For 2 addresses: LLLL=1, for 3 addresses: LLLL=2, etc.
                msgType = Math.min(Math.max(validAddresses.length - 1, 0), 15); // LLLL = NA (max 15)
                for (const addr of validAddresses.slice(0, 15)) {
                    addressBytes.push(...this.parseAddressDecimalOrHex(addr.trim()));
                }
                break;
        }
        // Build payload
        const payload = this.buildEverTagPayload(command, value);
        // Calculate total length (excluding CONFIG_ID, SEQ, LENGTH)
        const length = 1 + addressBytes.length + payload.length; // MSG_TYPE + addresses + payload
        packet.push(length & 0xFF);
        packet.push(msgType);
        packet.push(...addressBytes);
        packet.push(...payload);
        // Join bytes with a single space, no leading/trailing/double spaces
        const hexString = packet.map(b => b.toString(16).padStart(2, '0').toUpperCase()).join(' ');
        console.log('[UnifiedPacket] Final hex:', hexString);
        return hexString;
    },

    /**
     * Unified payload builder for EverTag config/app config
     * @param {object} command - Command object
     * @param {string} value - User input value
     * @returns {Array<number>} - Payload bytes
     */
    buildEverTagPayload: function(command, value) {
        const payload = [command.id];
        if (!command.valueType || command.valueType === 'none' || value === undefined || value === null || value === '') {
            return payload;
        }
        const val = value.trim();
        switch (command.valueType) {
            case 'byte': {
                let byteValue = parseInt(val);
                if (command.multiplier) {
                    byteValue = Math.floor(byteValue / command.multiplier);
                }
                if (byteValue < 0) {
                    byteValue = (byteValue + 256) % 256;
                }
                payload.push(byteValue & 0xFF);
                break;
            }
            case 'word': {
                const wordValue = parseInt(val);
                payload.push(wordValue & 0xFF);
                payload.push((wordValue >> 8) & 0xFF);
                break;
            }
            case 'dword': {
                const dwordValue = parseInt(val);
                payload.push(dwordValue & 0xFF);
                payload.push((dwordValue >> 8) & 0xFF);
                payload.push((dwordValue >> 16) & 0xFF);
                payload.push((dwordValue >> 24) & 0xFF);
                break;
            }
            case 'uuid': {
                // Remove dashes and validate UUID format
                const uuid = val.replace(/-/g, '').replace(/[^0-9a-fA-F]/g, '');
                
                // Ensure we have exactly 32 hex characters (16 bytes)
                if (uuid.length !== 32) {
                    console.warn(`Invalid UUID length: ${uuid.length} characters, expected 32`);
                }
                
                // Convert to 16 bytes, padding if necessary
                const paddedUuid = uuid.padEnd(32, '0');
                for (let i = 0; i < 32; i += 2) {
                    payload.push(parseInt(paddedUuid.substr(i, 2), 16));
                }
                break;
            }
            case 'nfc': {
                // Clean input to only digits and pad to 8 digits
                const pin = val.replace(/[^0-9]/g, '').padEnd(8, '0');
                
                console.log(`[NFC PIN] Input: "${val}" → Cleaned: "${pin}"`);
                
                // Convert each digit to its numeric value (1→0x01, 2→0x02, etc.)
                // and store in reverse order (little-endian)
                const bytes = [];
                for (let i = 0; i < 8; i++) {
                    const digit = parseInt(pin[i], 10);
                    bytes.push(digit & 0xFF);
                }
                
                // Reverse the byte order (little-endian format)
                // Example: 12345678 → [0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08] → [0x08,0x07,0x06,0x05,0x04,0x03,0x02,0x01]
                bytes.reverse();
                
                console.log(`[NFC PIN] Digits: [${pin.split('').join(', ')}] → Bytes: [${bytes.map(b => '0x' + b.toString(16).padStart(2, '0').toUpperCase()).join(', ')}]`);
                
                payload.push(...bytes);
                break;
            }
            case 'ledpattern': {
                const parts = val.split(',').map(p => p.trim());
                
                // First part is repeat count
                const repeatCount = parseInt(parts[0] || '0', 10) & 0xFF;
                payload.push(repeatCount);
                
                // Next 3 parts are pattern bytes 
                // Each pattern byte contains exactly 2 blink codes (0, 1, or 2)
                // Format: AB where A and B are single digits (0, 1, or 2)
                for (let i = 1; i <= 3; i++) {
                    if (i < parts.length && parts[i]) {
                        const patternStr = parts[i].replace(/[^0-2]/g, ''); // Only allow 0, 1, 2
                        let digits = patternStr.substring(0, 2).padEnd(2, '0'); // Take exactly 2 digits, pad if needed
                        
                        // Convert two-digit string directly to hex byte
                        // Example: "21" -> 0x21, "12" -> 0x12, "00" -> 0x00
                        const patternByte = parseInt(digits, 16);
                        
                        // Validate that each digit is 0, 1, or 2
                        const digit1 = parseInt(digits[0]);
                        const digit2 = parseInt(digits[1]);
                        if (digit1 <= 2 && digit2 <= 2) {
                            payload.push(patternByte);
                        } else {
                            payload.push(0); // Invalid digits, use 0x00
                        }
                    } else {
                        payload.push(0); // Default pattern byte (0x00 = no blinks)
                    }
                }
                break;
            }
            
            case 'pattern': {
                const parts = val.split(',').map(p => parseInt(p.trim(), 10));
                for (let i = 0; i < 4 && i < parts.length; i++) {
                    payload.push((parts[i] || 0) & 0xFF);
                }
                break;
            }
            case 'ibeacon_major_minor': {
                // Parse Major,Minor as decimal values and combine into 4-byte format
                // Format: Major 0xAABB, Minor 0xCCDD → sent as little-endian bytes
                const parts = val.split(',').map(p => parseInt(p.trim(), 10));
                const major = (parts[0] || 0) & 0xFFFF; // 16-bit major
                const minor = (parts[1] || 0) & 0xFFFF; // 16-bit minor
                
                console.log(`[iBeacon Major/Minor] Input: "${val}" → Major: ${major} (0x${major.toString(16).toUpperCase().padStart(4, '0')}), Minor: ${minor} (0x${minor.toString(16).toUpperCase().padStart(4, '0')})`);
                
                // Store as little-endian: Major low byte, Major high byte, Minor low byte, Minor high byte
                // Example: Major=234 (0x00EA), Minor=0 (0x0000) → [0xEA, 0x00, 0x00, 0x00]
                payload.push(major & 0xFF);         // Major low byte (EA)
                payload.push((major >> 8) & 0xFF);  // Major high byte (00)
                payload.push(minor & 0xFF);         // Minor low byte (00)
                payload.push((minor >> 8) & 0xFF);  // Minor high byte (00)
                
                console.log(`[iBeacon Major/Minor] Payload bytes (little-endian): [0x${(major & 0xFF).toString(16).toUpperCase().padStart(2, '0')}, 0x${((major >> 8) & 0xFF).toString(16).toUpperCase().padStart(2, '0')}, 0x${(minor & 0xFF).toString(16).toUpperCase().padStart(2, '0')}, 0x${((minor >> 8) & 0xFF).toString(16).toUpperCase().padStart(2, '0')}]`);
                break;
            }
            default: {
                // Always interpret as hex string for app config data
                const hexClean = val.replace(/[^0-9a-fA-F]/g, '');
                for (let i = 0; i < hexClean.length; i += 2) {
                    payload.push(parseInt(hexClean.substr(i, 2), 16));
                }
                break;
            }
        }
        return payload;
    },

    // Deprecated: Use buildEverTagPacket instead
    buildPacket: function() {
        console.warn('buildPacket is deprecated. Use buildEverTagPacket instead.');
        if (!this.selectedCommand) return '';
        return this.buildEverTagPacket(
            this.selectedCommand,
            this.commandValue,
            this.targetType,
            this.addresses,
            this.sequenceNumber
        );
    },

    /**
     * Update the packet display
     */
    updatePacket: function() {
        const packet = this.buildPacket();
        document.getElementById('raw_packet_output').value = packet;
    },

    /**
     * Update the send button state
     */
    updateSendButton: function() {
        const btn = document.getElementById('send_config_btn');
        if (!btn) return; // Button might not exist on all pages
        
        // Check basic requirements
        if (!this.selectedCommand) {
            btn.disabled = true;
            return;
        }
        
        // Check value validation
        const validation = this.validateCommandValue(this.selectedCommand, this.commandValue);
        const hasValidValue = this.selectedCommand.valueType === 'none' || validation.isValid;
        
        // Check address requirements
        const hasValidAddresses = this.targetType === 'broadcast-tags' || 
                                 this.targetType === 'broadcast-anchors' || 
                                 this.addresses.every(addr => addr.trim());
        
        btn.disabled = !hasValidValue || !hasValidAddresses;
    },

    /**
     * Send the EverTag configuration command
     */
    sendConfigCommand: function() {
        try {
            // Get the generated packet
            const packet = this.buildPacket();
            if (!packet) {
                this.showStatus('Error: Unable to generate packet', 'error');
                return;
            }
            
            // Get gateway and sink from the send app message form as default
            const gatewayId = document.getElementById('gw_id_downlink')?.value || '';
            const sinkId = document.getElementById('sink_id_downlink')?.value || '';
            
            if (!gatewayId || !sinkId) {
                this.showStatus('Error: Please set Gateway and Sink in the Send app message section first', 'error');
                return;
            }
            
            // EverTag config commands use endpoint 130 by default
            const srcEp = 130;
            const destEp = 130;
            const qos = 0; // Normal QoS
            
            // Determine destination address based on target type
            let destAddress;
            switch (this.targetType) {
                case 'broadcast-tags':
                case 'broadcast-anchors':
                    destAddress = 0xFFFFFFFF; // Broadcast address
                    break;
                case 'unicast':
                    if (this.addresses.length > 0 && this.addresses[0]) {
                        // For the downlink destination, we need the full 32-bit address as integer
                        const addressStr = this.addresses[0].trim();
                        if (/[a-fA-F]/.test(addressStr) || addressStr.toLowerCase().startsWith('0x')) {
                            // Hex address
                            destAddress = parseInt(addressStr.replace(/[^0-9a-fA-F]/g, ''), 16);
                        } else {
                            // Decimal address
                            destAddress = parseInt(addressStr, 10);
                        }
                    } else {
                        this.showStatus('Error: Please specify target address', 'error');
                        return;
                    }
                    break;
                case 'multicast':
                    // For multicast, addresses are embedded in the packet, use broadcast
                    destAddress = 0xFFFFFFFF;
                    break;
                default:
                    this.showStatus('Error: Invalid target type', 'error');
                    return;
            }
            
            // Format packet as space-separated hex pairs
            const formattedPacket = packet.match(/.{2}/g).join(' ');
            
            // Send via the messaging system
            if (typeof uiMessages !== 'undefined' && uiMessages.sendMessage) {
                // Use the existing send message functionality
                const messageData = {
                    gw_id: gatewayId,
                    sink_id: sinkId,
                    dest_add: destAddress,
                    src_ep: srcEp,
                    dest_ep: destEp,
                    qos: qos,
                    payload: formattedPacket
                };
                
                // Populate the downlink form and send
                document.getElementById('gw_id_downlink').value = gatewayId;
                document.getElementById('sink_id_downlink').value = sinkId;
                document.getElementById('dest_add_downlink').value = destAddress;
                document.getElementById('src_ep_downlink').value = srcEp;
                document.getElementById('dest_ep_downlink').value = destEp;
                document.getElementById('qos_downlink').value = qos;
                document.getElementById('payload_downlink').value = formattedPacket;

                // Set pending sink selection so config response will trigger auto-population
                // Try to get networkId from the UI if available
                let networkId = '';
                const networkIdInput = document.getElementById('evertag_network_id');
                if (networkIdInput) {
                    networkId = networkIdInput.value.trim();
                }
                this.pendingSinkSelection = {
                    gatewayId: gatewayId,
                    networkId: networkId,
                    sinkId: sinkId
                };
                
                // Trigger the send
                uiMessages.onSentPressed();
                
                // Increment sequence number and save
                this.sequenceNumber = (this.sequenceNumber + 1) % 256;
                localStorage.setItem('evertagConfigSequence', this.sequenceNumber.toString());
                
                // Update sequence display
                const seqElement = document.getElementById('sequence_number');
                if (seqElement) {
                    seqElement.textContent = this.sequenceNumber;
                }
                
                // Update packet display with new sequence
                this.updatePacket();
                
                // Increment app config sequence number and save
                this.appConfigSequenceNumber = (this.appConfigSequenceNumber + 1) % 256;
                localStorage.setItem('appConfigSequence', this.appConfigSequenceNumber.toString());
                
                this.showStatus(`EverTag config sent: ${this.selectedCommand.name}`, 'success');
                
            } else {
                this.showStatus('Error: Messaging system not available', 'error');
            }
            
        } catch (error) {
            console.error('Error sending EverTag config:', error);
            this.showStatus('Error: Failed to send configuration', 'error');
        }
    },

    /**
     * Show status message
     */
    showStatus: function(message, type) {
        // Try to find status elements - they might not exist in the messages page context
        const statusDiv = document.getElementById('status_message');
        const statusText = document.getElementById('status_text');
        
        if (!statusDiv || !statusText) {
            // If status elements don't exist, just log to console
            console.log(`Status (${type}): ${message}`);
            return;
        }
        
        // Remove existing classes
        statusDiv.className = 'w3-panel';
        
        // Add appropriate class based on type
        switch (type) {
            case 'success':
                statusDiv.classList.add('w3-pale-green', 'w3-border-green');
                break;
            case 'error':
                statusDiv.classList.add('w3-pale-red', 'w3-border-red');
                break;
            case 'info':
                statusDiv.classList.add('w3-pale-blue', 'w3-border-blue');
                break;
        }
        
        statusText.textContent = message;
        statusDiv.classList.remove('w3-hide');
        
        // Auto-hide after 5 seconds for success/info messages
        if (type !== 'error') {
            setTimeout(() => {
                statusDiv.classList.add('w3-hide');
            }, 5000);
        }
    },

    /**
     * Convert hex string to bytes array (handles both spaced and non-spaced hex)
     * @param {string} hexString - Hex string (e.g., "B0 0B 02 40 12" or "B00B024012")
     * @returns {Array<number>} - Array of byte values
     */
    hexToBytes: function(hexString) {
        if (!hexString || typeof hexString !== 'string') {
            return [];
        }
        
        // Remove all non-hex characters (spaces, etc.) and ensure even length
        const cleaned = hexString.replace(/[^0-9a-fA-F]/g, '');
        if (cleaned.length % 2 !== 0) {
            throw new Error('Invalid hex string: odd number of characters');
        }
        
        const bytes = [];
        for (let i = 0; i < cleaned.length; i += 2) {
            const hexByte = cleaned.substr(i, 2);
            const byteValue = parseInt(hexByte, 16);
            if (isNaN(byteValue)) {
                throw new Error(`Invalid hex byte: ${hexByte}`);
            }
            bytes.push(byteValue);
        }
        
        return bytes;
    },

    /**
     * Update Application and Diagnostics Configuration (like the reference implementation)
     * This sends Wirepas diagnostics interval configuration to the network
     */
    updateAppConfig: function() {
        try {
            // Get values from the App config wizard section
            const networkId = document.getElementById('evertag_network_id').value.trim();
            const gatewayId = document.getElementById('evertag_gateway_id').value.trim();
            const sinkId = document.getElementById('evertag_sink_id').value.trim();
            const diagInterval = document.getElementById('evertag_diag_interval').value;
            const appConfigData = document.getElementById('evertag_app_config_data').value;
            
            // Validate inputs
            if (!networkId) {
                this.showStatus('Error: Please select a network', 'error');
                return;
            }
            
            if (!gatewayId) {
                this.showStatus('Error: Please select a gateway', 'error');
                return;
            }
            
            if (!sinkId) {
                this.showStatus('Error: Please select a sink', 'error');
                return;
            }
            
            // Parse hex data if provided - use our local hexToBytes function
            let appConfigBytes = [];
            if (appConfigData.trim()) {
                try {
                    appConfigBytes = this.hexToBytes(appConfigData.trim());
                } catch (error) {
                    this.showStatus('Error: Invalid hex data format', 'error');
                    return;
                }
            }
            
            // Build app config object like the reference implementation
            const appConfig = {
                app_config: {
                    seq: this.tagConfigSequenceNumber,  // Use tagConfigSequenceNumber for app config
                    diag_interval_s: parseInt(diagInterval, 10),
                    app_config_data: appConfigBytes
                },
                network_address: parseInt(networkId, 10),
                sink_id: sinkId
            };
            
            this.showStatus('Sending Wirepas diagnostics configuration...', 'info');
            
            // Log the configuration for debugging
            console.log('EverTag App Config:', appConfig);
            console.log('Network ID:', networkId, 'Gateway ID:', gatewayId, 'Sink ID:', sinkId);
            
            // Call networkInterface.setSinkConfig() like the reference implementation
            if (typeof networkInterface !== 'undefined' && networkInterface.setSinkConfig) {
                networkInterface.setSinkConfig(gatewayId, sinkId, appConfig);
                
                // Auto-increment sequence number after successful send (if auto-increment is enabled)
                // Try multiple possible checkbox IDs to find the correct one
                let autoIncrementCheckbox = document.getElementById('auto_increment_tag_sequence') ||
                                          document.getElementById('auto_increment_after_send') ||
                                          document.querySelector('input[type="checkbox"]:checked') ||
                                          document.querySelector('input[type="checkbox"]');
                
                // Debug: Log which checkbox elements we found
                console.log('Debug: Looking for auto-increment checkbox...');
                console.log('Debug: Found auto_increment_tag_sequence:', document.getElementById('auto_increment_tag_sequence'));
                console.log('Debug: Found auto_increment_after_send:', document.getElementById('auto_increment_after_send'));
                console.log('Debug: Found any checked checkbox:', document.querySelector('input[type="checkbox"]:checked'));
                
                // Find the checkbox that is actually checked and relates to auto-increment
                const allCheckboxes = document.querySelectorAll('input[type="checkbox"]');
                console.log('Debug: All checkboxes found:', allCheckboxes.length);
                for (let i = 0; i < allCheckboxes.length; i++) {
                    const cb = allCheckboxes[i];
                    console.log(`Debug: Checkbox ${i}: ID="${cb.id}", checked=${cb.checked}, name="${cb.name}"`);
                    if (cb.checked && (cb.id.includes('auto') || cb.id.includes('increment'))) {
                        autoIncrementCheckbox = cb;
                        console.log('Debug: Using checkbox with ID:', cb.id);
                        break;
                    }
                }
                
                if (autoIncrementCheckbox && autoIncrementCheckbox.checked) {
                    console.log('Debug: Auto-incrementing sequence number from', this.tagConfigSequenceNumber);
                    this.tagConfigSequenceNumber = (this.tagConfigSequenceNumber + 1) % 256;
                    localStorage.setItem('tagConfigSequence', this.tagConfigSequenceNumber.toString());
                    console.log('Debug: New sequence number:', this.tagConfigSequenceNumber);
                    
                    // Update the input field to show new sequence number
                    const tagSeqInput = document.getElementById('tag_config_sequence');
                    if (tagSeqInput) {
                        tagSeqInput.value = this.tagConfigSequenceNumber;
                    }
                    
                    // Update the app config tag sequence input field
                    const appConfigTagSeqInput = document.getElementById('appconfig_tag_config_sequence');
                    if (appConfigTagSeqInput) {
                        appConfigTagSeqInput.value = this.tagConfigSequenceNumber;
                    }
                    
                    // Update packet display with new sequence number
                    this.updateAppConfigPacketDisplay();
                } else {
                    console.log('Debug: Auto-increment checkbox not found or not checked');
                }
                
                this.showStatus('Wirepas diagnostics configuration sent successfully!', 'success');
            } else {
                this.showStatus('Error: Network interface not available', 'error');
            }
            
        } catch (error) {
            console.error('Error updating app config:', error);
            this.showStatus('Error updating app config: ' + error.message, 'error');
        }
    },

    /**
     * Handle network selection change
     */
    onNetworkChanged: function(networkId) {
        const gatewaySelect = document.getElementById('evertag_gateway_id');
        const sinkSelect = document.getElementById('evertag_sink_id');
        
        // Clear gateway and sink dropdowns
        gatewaySelect.innerHTML = '<option value="">-- Select Gateway --</option>';
        sinkSelect.innerHTML = '<option value="">-- Select Sink --</option>';
        
        if (!networkId) {
            return;
        }
        
        // Populate gateways for the selected network
        this.populateGatewaysForNetwork(networkId);

        // Save the selected network (only if not restoring)
        if (!this.isRestoringSelections) {
            this.cachedSelections.networkId = networkId;
            // Clear dependent selections when user manually changes network
            this.cachedSelections.gatewayId = null;
            this.cachedSelections.sinkId = null;
            localStorage.setItem('evertagCachedSelections', JSON.stringify(this.cachedSelections));
            console.log('Saved network selection:', networkId);
        }
    },

    /**
     * Handle gateway selection change
     */
    onGatewayChanged: function(gatewayId) {
        const sinkSelect = document.getElementById('evertag_sink_id');
        const networkId = document.getElementById('evertag_network_id').value;
        
        // Clear sink dropdown
        sinkSelect.innerHTML = '<option value="">-- Select Sink --</option>';
        
        if (!gatewayId || !networkId) {
            return;
        }
        
        // Populate sinks for the selected gateway and network
        this.populateSinksForGateway(gatewayId, networkId);

        // Save the selected gateway (only if not restoring)
        if (!this.isRestoringSelections) {
            this.cachedSelections.gatewayId = gatewayId;
            // Clear sink selection when user manually changes gateway
            this.cachedSelections.sinkId = null;
            localStorage.setItem('evertagCachedSelections', JSON.stringify(this.cachedSelections));
            console.log('Saved gateway selection:', gatewayId);
        }
    },

    /**
     * Populate gateways for a specific network
     */
    populateGatewaysForNetwork: function(networkId) {
        try {
            const gatewaySelect = document.getElementById('evertag_gateway_id');
            if (!gatewaySelect) return;
            
            console.log('Debug: Populating gateways for network:', networkId);
            
            // Get gateways that have configurations for this network
            const availableGateways = new Set();
            
            // Method 1: Try from gateway configurations
            if (typeof mainApp !== 'undefined' && mainApp.getAllGateways) {
                const gateways = mainApp.getAllGateways();
                console.log('Debug: All gateways:', gateways);
                
                if (gateways && typeof gateways === 'object') {
                    Object.entries(gateways).forEach(([gatewayId, gateway]) => {
                        console.log(`Debug: Checking gateway ${gatewayId}:`, gateway);
                        
                        if (gateway.configs && Array.isArray(gateway.configs)) {
                            gateway.configs.forEach((config, index) => {
                                console.log(`Debug: Gateway ${gatewayId} config ${index}:`, config);
                                console.log(`Debug: Network address: ${config.network_address}, Looking for: ${networkId}`);
                                
                                // Try both string and number comparison
                                if (config.network_address == networkId || String(config.network_address) === String(networkId)) {
                                    availableGateways.add(gatewayId);
                                    console.log(`Debug: Added gateway ${gatewayId} for network ${networkId}`);
                                }
                            });
                        } else {
                            console.log(`Debug: Gateway ${gatewayId} has no configs or configs is not array`);
                        }
                    });
                }
            } else {
                console.log('Debug: mainApp or getAllGateways not available');
            }
            
            // Method 2: If no gateways found from configs, try from messages
            if (availableGateways.size === 0) {
                console.log('Debug: No gateways found from configs, trying from messages...');
                
                if (typeof mainApp !== 'undefined' && mainApp.getAllMessages) {
                    const messages = mainApp.getAllMessages();
                    if (Array.isArray(messages)) {
                        messages.forEach((message, index) => {
                            // Log first few messages for debugging
                            if (index < 3) {
                                console.log(`Debug: Message ${index}:`, message);
                            }
                            
                            // Check if message is from the selected network
                            if ((message.network_id == networkId || String(message.network_id) === String(networkId)) && message.gw_id) {
                                availableGateways.add(message.gw_id);
                                if (index < 10) { // Log first 10 matches
                                    console.log(`Debug: Added gateway ${message.gw_id} from message for network ${networkId}`);
                                }
                            }
                        });
                    }
                }
            }
            
            console.log('Debug: Available gateways for network', networkId, ':', Array.from(availableGateways));
            
            // Populate gateway dropdown
            const sortedGateways = Array.from(availableGateways).sort();
            sortedGateways.forEach(gatewayId => {
                const option = document.createElement('option');
                option.value = gatewayId;
                option.textContent = `Gateway ${gatewayId}`;
                gatewaySelect.appendChild(option);
            });
            
            if (sortedGateways.length === 0) {
                console.log('Debug: No gateways found for network', networkId);
            }
            
        } catch (error) {
            console.error('Error populating gateways dropdown:', error);
        }
    },

    /**
     * Populate sinks for a specific gateway and network
     */
    populateSinksForGateway: function(gatewayId, networkId) {
        try {
            const sinkSelect = document.getElementById('evertag_sink_id');
            if (!sinkSelect) return;
            
            console.log('Debug: Populating sinks for gateway:', gatewayId, 'network:', networkId);
            
            // Get sinks for the selected gateway and network
            const availableSinks = new Set();
            
            // Method 1: Try from gateway configurations
            if (typeof mainApp !== 'undefined' && mainApp.getAllGateways) {
                const gateways = mainApp.getAllGateways();
                if (gateways && gateways[gatewayId]) {
                    const gateway = gateways[gatewayId];
                    console.log('Debug: Gateway data:', gateway);
                    
                    if (gateway.configs && Array.isArray(gateway.configs)) {
                        gateway.configs.forEach((config, index) => {
                            console.log(`Debug: Gateway ${gatewayId} config ${index}:`, config);
                            console.log(`Debug: Network address: ${config.network_address}, sink_id: ${config.sink_id}`);
                            
                            // Try both string and number comparison for network
                            if ((config.network_address == networkId || String(config.network_address) === String(networkId)) && config.sink_id) {
                                availableSinks.add(config.sink_id);
                                console.log(`Debug: Added sink ${config.sink_id} for gateway ${gatewayId} network ${networkId}`);
                            }
                        });
                    } else {
                        console.log(`Debug: Gateway ${gatewayId} has no configs or configs is not array`);
                    }
                } else {
                    console.log(`Debug: Gateway ${gatewayId} not found in gateways data`);
                }
            } else {
                console.log('Debug: mainApp or getAllGateways not available');
            }
            
            // Method 2: If no sinks found from configs, try from messages
            if (availableSinks.size === 0) {
                console.log('Debug: No sinks found from configs, trying from messages...');
                
                if (typeof mainApp !== 'undefined' && mainApp.getAllMessages) {
                    const messages = mainApp.getAllMessages();
                    if (Array.isArray(messages)) {
                        messages.forEach((message, index) => {
                            // Check if message is from the selected network and gateway
                            if ((message.network_id == networkId || String(message.network_id) === String(networkId)) && 
                                (message.gw_id == gatewayId || String(message.gw_id) === String(gatewayId)) && 
                                message.sink_id) {
                                availableSinks.add(message.sink_id);
                                if (index < 10) { // Log first 10 matches
                                    console.log(`Debug: Added sink ${message.sink_id} from message for gateway ${gatewayId} network ${networkId}`);
                                }
                            }
                        });
                    }
                }
            }
            
            console.log('Debug: Available sinks for gateway', gatewayId, 'network', networkId, ':', Array.from(availableSinks));
            
            // Populate sink dropdown
            const sortedSinks = Array.from(availableSinks).sort();
            sortedSinks.forEach(sinkId => {
                const option = document.createElement('option');
                option.value = sinkId;
                option.textContent = `Sink ${sinkId}`;
                sinkSelect.appendChild(option);
            });
            
            // If there's only one sink, auto-select it and populate config data
            if (sortedSinks.length === 1) {
                sinkSelect.value = sortedSinks[0];
                console.log('Debug: Auto-selecting single sink:', sortedSinks[0]);
                // Trigger auto-population for the single sink using the same method
                this.onSinkChanged(sortedSinks[0]);
            }
            
            if (sortedSinks.length === 0) {
                console.log('Debug: No sinks found for gateway', gatewayId, 'network', networkId);
            }
            
        } catch (error) {
            console.error('Error populating sinks dropdown:', error);
        }
    },

    /**
     * Handle sink selection change and auto-populate diagnostics and data fields
     * Now checks if user has manually entered data to avoid overwriting user input
     */
    onSinkChanged: function(sinkId) {
        const networkId = document.getElementById('evertag_network_id').value;
        const gatewayId = document.getElementById('evertag_gateway_id').value;
        
        // Don't auto-populate if user is actively editing the app config data field
        const appConfigDataField = document.getElementById('evertag_app_config_data');
        const diagIntervalField = document.getElementById('evertag_diag_interval');
        
        // Check if user has manually entered data
        const hasUserData = (appConfigDataField && appConfigDataField.value.trim() && 
                           !appConfigDataField.value.includes('B0')) || // Not just our generated packet
                          (diagIntervalField && diagIntervalField.value !== '0' && diagIntervalField.value !== '');
        
        if (hasUserData) {
            console.log('Debug: User has entered app config data, skipping auto-population');
            return;
        }
        
        // Clear the fields first
        if (diagIntervalField) diagIntervalField.value = '0';
        if (appConfigDataField) appConfigDataField.value = '';
        
        if (!sinkId || !networkId || !gatewayId) {
            return;
        }
        
        // Request current configuration from gateway
        console.log('Debug: Requesting current config for gateway:', gatewayId);
        if (typeof mainApp !== 'undefined' && mainApp.askGatewayConfig) {
            mainApp.askGatewayConfig(gatewayId);
            this.showStatus('Requesting current configuration from gateway...', 'info');
            
            // Store the selected sink info for when the config response arrives
            this.pendingSinkSelection = {
                gatewayId: gatewayId,
                networkId: networkId,
                sinkId: sinkId
            };
            console.log('Debug: Set pending sink selection:', this.pendingSinkSelection);
        } else {
            this.showStatus('Error: Unable to request gateway configuration', 'error');
        }

        // Save the selected sink (only if not restoring)
        if (!this.isRestoringSelections) {
            this.cachedSelections.sinkId = sinkId;
            localStorage.setItem('evertagCachedSelections', JSON.stringify(this.cachedSelections));
            console.log('Saved sink selection:', sinkId);
        }
    },

    /**
     * Called when gateway configuration is received (hook into the gateway config response)
     * This mimics how the gateway page auto-populates fields when config is received
     */
    onGatewayConfigReceived: function(gatewayId) {
        // Check if we have a pending sink selection for this gateway
        if (
            this.pendingSinkSelection &&
            this.pendingSinkSelection.gatewayId &&
            gatewayId &&
            this.pendingSinkSelection.gatewayId === gatewayId
        ) {
            console.log('Debug: Gateway config received for pending sink selection');
            // Now populate the fields using the fresh config data
            setTimeout(() => {
                try {
                    this.populateSinkConfigData(
                        this.pendingSinkSelection.gatewayId,
                        this.pendingSinkSelection.networkId,
                        this.pendingSinkSelection.sinkId
                    );
                    this.pendingSinkSelection = null; // Clear pending selection
                } catch (error) {
                    console.error('Error in onGatewayConfigReceived:', error);
                    this.pendingSinkSelection = null; // Clear on error
                }
            }, 500); // Increased delay to ensure config is fully processed and DOM is ready
        }
    },

    /**
     * Populate diagnostics interval and data fields from current sink configuration
     * This is called after fresh config data has been requested from the gateway
     */
    populateSinkConfigData: function(gatewayId, networkId, sinkId) {
        try {
            console.log('Debug: Auto-populating config data for gateway:', gatewayId, 'network:', networkId, 'sink:', sinkId);
            
            // Get current sink configuration from fresh gateway data (like gateway page does)
            let sinkConfig = null;
            
            if (typeof mainApp !== 'undefined' && mainApp.getAllGateways) {
                const gateways = mainApp.getAllGateways();
                if (gateways && gateways[gatewayId]) {
                    const gateway = gateways[gatewayId];
                    console.log('Debug: Found gateway with fresh config:', gateway);
                    
                    if (gateway.configs && Array.isArray(gateway.configs)) {
                        // Find the sink configuration that matches both network and sink ID
                        gateway.configs.forEach((config, index) => {
                            console.log(`Debug: Checking config ${index}:`, config);
                            
                            if ((config.network_address == networkId || String(config.network_address) === String(networkId)) && 
                                (config.sink_id == sinkId || String(config.sink_id) === String(sinkId))) {
                                sinkConfig = config;
                                console.log('Debug: Found matching sink config:', sinkConfig);
                            }
                        });
                    }
                }
            }
            
            if (sinkConfig && sinkConfig.app_config) {
                console.log('Debug: Auto-populating from fresh app_config:', sinkConfig.app_config);
                
                // Auto-populate diagnostics interval (same as gateway page)
                if (sinkConfig.app_config.diag_interval_s !== undefined) {
                    const diagInterval = sinkConfig.app_config.diag_interval_s;
                    const diagSelect = document.getElementById('evertag_diag_interval');
                    if (diagSelect) {
                        diagSelect.value = diagInterval;
                        console.log('Debug: Set diagnostics interval to:', diagInterval);
                    }
                }
                
                // Auto-populate app config data (convert bytes to hex string like gateway page)
                if (sinkConfig.app_config.app_config_data && 
                    (Array.isArray(sinkConfig.app_config.app_config_data) || 
                     sinkConfig.app_config.app_config_data instanceof Uint8Array)) {
                    let hexString = '';
                    // Handle both regular arrays and Uint8Arrays
                    const dataArray = sinkConfig.app_config.app_config_data;
                    for (let i = 0; i < dataArray.length; i++) {
                        hexString += ('0' + (dataArray[i] & 0xff).toString(16)).slice(-2);
                    }
                    const finalHex = hexString.toUpperCase();
                    const dataField = document.getElementById('evertag_app_config_data');
                    if (dataField) {
                        dataField.value = finalHex;
                        console.log('Debug: Set app config data to:', finalHex);
                        console.log('Debug: Original data array type:', dataArray.constructor.name);
                        console.log('Debug: Original data array length:', dataArray.length);
                    } else {
                        console.error('Debug: Could not find evertag_app_config_data element');
                    }
                } else {
                    console.log('Debug: No app_config_data found or not an array/Uint8Array:', sinkConfig.app_config.app_config_data);
                    const dataField = document.getElementById('evertag_app_config_data');
                    if (dataField) {
                        dataField.value = '';
                    }
                }
                
                this.showStatus('Configuration auto-populated from current sink settings', 'success');
            } else {
                console.log('Debug: No app_config found for sink after requesting fresh config');
                this.showStatus('No existing configuration found, using defaults', 'info');
            }
            
        } catch (error) {
            console.error('Error auto-populating sink config data:', error);
            this.showStatus('Error loading current configuration: ' + error.message, 'error');
        }
    },

    /**
     * Force population of networks (called when needed)
     */
    forcePopulateNetworks: function() {
        console.log('Force populating networks...');
        
        // Clear any existing interval
        if (this.liveUpdateInterval) {
            clearInterval(this.liveUpdateInterval);
            this.liveUpdateInterval = null;
        }
        
        // Populate immediately
        this.populateNetworksDropdown();
        
        // Restore cached selections after networks are populated
        this.restoreCachedSelections();
        
        // Restart live update if it was enabled
        const liveUpdateCheckbox = document.getElementById('live_update_evertag');
        if (liveUpdateCheckbox && liveUpdateCheckbox.checked) {
            this.liveUpdateEnabled = true;
            this.liveUpdateInterval = setInterval(() => {
                if (this.liveUpdateEnabled) {
                    this.populateNetworksDropdown();
                }
            }, 3000);
        }
    },

    /**
     * Load app config commands
     */
    loadAppConfigCommands: function() {
        const select = document.getElementById('appconfig_evertag_command_select');
        if (!select) {
            console.log('Debug: appconfig_evertag_command_select not found');
            return;
        }
        
        console.log('Debug: Loading app config commands...');
        
        // Clear existing options except the first one
        while (select.children.length > 1) {
            select.removeChild(select.lastChild);
        }
        
        // Add all available commands
        this.commands.forEach((command, index) => {
            const option = document.createElement('option');
            option.value = index; // Use array index as value
            option.textContent = `0x${command.id.toString(16).toUpperCase().padStart(2, '0')} - ${command.name}`;
            select.appendChild(option);
        });
        
        console.log('Debug: Added', this.commands.length, 'commands to dropdown');
    },

    /**
     * Handle app config command selection change
     */
    onAppConfigCommandChanged: function(commandIndex) {
        console.log('Debug: App config command changed:', commandIndex);
        this.appConfigCommand = commandIndex !== '' ? this.commands[parseInt(commandIndex)] : null;
        
        const valueSection = document.getElementById('appconfig_command_value_section');
        const valueInput = document.getElementById('appconfig_command_value_input');
        const descSection = document.getElementById('appconfig_command_description');
        const descText = document.getElementById('appconfig_command_description_text');
        
        if (this.appConfigCommand) {
            // Show/hide value input based on command requirements
            if (this.appConfigCommand.valueType && this.appConfigCommand.valueType !== 'none') {
                valueSection.classList.remove('w3-hide');
                
                // Clear all attributes first to prevent carryover from previous commands
                valueInput.removeAttribute('min');
                valueInput.removeAttribute('max');
                valueInput.removeAttribute('maxLength');
                valueInput.removeAttribute('pattern');
                valueInput.removeAttribute('title');
                valueInput.placeholder = '';
                
                // Set input type and constraints based on command
                switch (this.appConfigCommand.valueType) {
                    case 'byte':
                    case 'word':
                    case 'dword':
                        valueInput.type = 'number';
                        // Apply multiplier to get the actual user input range
                        const actualMin = this.appConfigCommand.multiplier ? 
                            (this.appConfigCommand.min || 0) * this.appConfigCommand.multiplier : 
                            (this.appConfigCommand.min || 0);
                        const actualMax = this.appConfigCommand.multiplier ? 
                            (this.appConfigCommand.max || 255) * this.appConfigCommand.multiplier : 
                            (this.appConfigCommand.max || 255);
                        valueInput.min = actualMin;
                        valueInput.max = actualMax;
                        valueInput.title = `Range: ${actualMin || 'N/A'} - ${actualMax || 'N/A'} ${this.appConfigCommand.unit || ''}`;
                        break;
                    case 'ledpattern':
                        valueInput.type = 'text';
                        valueInput.placeholder = '5,21,12,21';
                        valueInput.title = 'LED Pattern: repeat_count,byte1,byte2,byte3 (digits 0,1,2 only)';
                        break;
                    case 'uuid':
                        valueInput.type = 'text';
                        valueInput.placeholder = 'Enter UUID or copy: F2746FD6-483D-11EE-BE56-0242AC120002';
                        valueInput.maxLength = 36; // Standard UUID format length
                        valueInput.pattern = '[0-9A-Fa-f]{8}-[0-9A-Fa-f]{4}-[0-9A-Fa-f]{4}-[0-9A-Fa-f]{4}-[0-9A-Fa-f]{12}'; // UUID pattern
                        valueInput.title = 'Enter a valid UUID in format: XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX';
                        break;
                    case 'nfc':
                        valueInput.type = 'text';
                        valueInput.placeholder = '12345678';
                        valueInput.maxLength = 8;
                        valueInput.title = '8-digit PIN (e.g., 12345678) - each digit becomes 0x01-0x09, sent in reverse order';
                        break;
                    case 'ibeacon_major_minor':
                        valueInput.type = 'text';
                        valueInput.placeholder = '234,112';
                        valueInput.title = 'Enter Major,Minor as decimal values (e.g., 234,112)';
                        break;
                    default:
                        valueInput.type = 'text';
                        valueInput.placeholder = this.appConfigCommand.description || '';
                        valueInput.title = this.appConfigCommand.description || 'Enter value for this command';
                        break;
                }
            } else {
                valueSection.classList.add('w3-hide');
                this.appConfigValue = '';
            }
            
            // Show command description
            descSection.classList.remove('w3-hide');
            descText.textContent = this.appConfigCommand.description || '';
        } else {
            valueSection.classList.add('w3-hide');
            descSection.classList.add('w3-hide');
            this.appConfigValue = '';
        }
        
        // Update target type visibility
        this.updateAppConfigTargetTypeVisibility();
        this.updateAppConfigPacketDisplay();
    },

    /**
     * Handle app config value change
     */
    onAppConfigValueChanged: function(value) {
        console.log('Debug: App config value changed:', value);
        this.appConfigValue = value;
        
        // Validate the input for app config
        if (this.appConfigCommand) {
            const validation = this.validateCommandValue(this.appConfigCommand, value);
            const input = document.getElementById('appconfig_command_value_input');
            
            // Show validation error if invalid
            if (!validation.isValid && value.trim()) {
                input.style.borderColor = '#f44336';
                input.title = validation.error;
                setTimeout(() => {
                    input.style.borderColor = '';
                    input.title = input.getAttribute('data-original-title') || '';
                }, 3000);
            } else {
                input.style.borderColor = '';
            }
        }
        
        this.updateAppConfigPacketDisplay();
    },

    /**
     * Handle app config target type change
     */
    onAppConfigTargetTypeChanged: function(targetType) {
        console.log('Debug: App config target type changed:', targetType);
        this.appConfigTargetType = targetType;
        this.updateAppConfigTargetTypeVisibility();
        this.updateAppConfigPacketDisplay();
    },

    /**
     * Handle app config tag sequence change
     */
    onAppConfigTagSequenceChanged: function(sequence) {
        console.log('Debug: App config tag sequence changed:', sequence);
        this.tagConfigSequenceNumber = parseInt(sequence) || 1;
        localStorage.setItem('tagConfigSequence', this.tagConfigSequenceNumber.toString());
        this.updateAppConfigPacketDisplay();
    },

    /**
     * Update target type visibility for app config
     */
    updateAppConfigTargetTypeVisibility: function() {
        const addressSection = document.getElementById('appconfig_address_section');
        const addBtn = document.getElementById('appconfig_add_address_btn');
        
        if (this.appConfigTargetType === 'unicast' || this.appConfigTargetType === 'multicast') {
            addressSection.classList.remove('w3-hide');
            
            if (this.appConfigTargetType === 'unicast') {
                this.appConfigAddresses = [''];
                addBtn.classList.add('w3-hide');
            } else {
                if (this.appConfigAddresses.length === 0) {
                    this.appConfigAddresses = ['', ''];
                }
                addBtn.classList.remove('w3-hide');
            }
            
            this.updateAppConfigAddressInputs();
        } else {
            addressSection.classList.add('w3-hide');
            this.appConfigAddresses = [];
        }
    },

    /**
     * Add address input for app config
     */
    addAppConfigAddress: function() {
        if (this.appConfigAddresses.length < 15) {
            this.appConfigAddresses.push('');
            this.updateAppConfigAddressInputs();
        }
    },

    /**
     * Update address inputs for app config
     */
    updateAppConfigAddressInputs: function() {
        const container = document.getElementById('appconfig_address_inputs');
        if (!container) return;
        
        container.innerHTML = '';
        
        this.appConfigAddresses.forEach((address, index) => {
            const wrapper = document.createElement('div');
            wrapper.style.display = 'inline-block';
            wrapper.style.marginRight = '10px';
            wrapper.style.marginBottom = '5px';
            
            const input = document.createElement('input');
            input.type = 'number';
            input.min = '0';
            input.max = '4294967295';
            input.value = address;
            input.style.width = '12ch';
            input.style.marginRight = '5px';
            input.placeholder = 'Node address';
            input.onchange = (e) => {
                this.appConfigAddresses[index] = e.target.value;
                this.updateAppConfigPacketDisplay();
            };
            
            wrapper.appendChild(input);
            
            // Add remove button for multicast (and for unicast if there are multiple somehow)
            if (this.appConfigTargetType === 'multicast' && this.appConfigAddresses.length > 1) {
                const removeBtn = document.createElement('button');
                removeBtn.type = 'button';
                removeBtn.className = 'w3-button w3-small w3-red';
                removeBtn.textContent = '×';
                removeBtn.onclick = () => {
                    this.appConfigAddresses.splice(index, 1);
                    this.updateAppConfigAddressInputs();
                    this.updateAppConfigPacketDisplay();
                };
                wrapper.appendChild(removeBtn);
            }
            
            container.appendChild(wrapper);
        });
    },

    /**
     * Update packet display for app config
     */
    updateAppConfigPacketDisplay: function() {
        // Update the hex data field based on current EverTag command settings
        if (this.appConfigCommand) {
            try {
                const hexString = this.buildEverTagPacket(
                    this.appConfigCommand,
                    this.appConfigValue,
                    this.appConfigTargetType,
                    this.appConfigAddresses,
                    this.tagConfigSequenceNumber
                );
                // hexString is already properly formatted with spaces, no need to re-format
                const dataField = document.getElementById('evertag_app_config_data');
                if (dataField) {
                    dataField.value = hexString;
                }
            } catch (error) {
                console.error('Error building app config packet:', error);
            }
        } else {
            // Clear the hex field if no command selected
            const dataField = document.getElementById('evertag_app_config_data');
            if (dataField) {
                dataField.value = '';
            }
        }
    }
};

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    // Delay initialization to ensure other UI components are loaded
    setTimeout(() => {
        // Check if we're on a page with Quick Diagnostics (Messages page)
        const quickDiagnosticsElement = document.querySelector('details summary b');
        const isMessagesPage = quickDiagnosticsElement && 
                               (quickDiagnosticsElement.textContent.includes('Send app config') ||
                                quickDiagnosticsElement.textContent.includes('EverTag config'));
        
        if (isMessagesPage) {
            console.log('Detected Messages page, using Messages-specific initialization');
            uiEverTagConfig.initializeForMessages();
        } else {
            console.log('Using standard EverTag initialization');
            uiEverTagConfig.initialize();
        }
    }, 1000);
    
    // Also add a backup initialization when the Messages page becomes visible
    setTimeout(() => {
        const messageMode = document.getElementById('message_mode');
        if (messageMode) {
            // Force populate networks when we first access the messages page
            uiEverTagConfig.forcePopulateNetworks();
        }
    }, 2000); // Longer delay to ensure all data is loaded
});

// Global helper function for debugging - can be called from browser console
window.forceNetworkRefresh = function() {
    console.log('Forcing network refresh from global function...');
    if (typeof uiEverTagConfig !== 'undefined') {
        uiEverTagConfig.forcePopulateNetworks();
    } else {
        console.error('uiEverTagConfig not available');
    }
}; 