/**
 * ui_evertag.js
 *
 * Adds EverTag-specific UI and logic for interacting with EverTag devices.
 * - Provides device-specific controls and data display.
 * - Integrates with mainApp and other modules for EverTag workflows.
 * - See README.md for extension patterns.
 */

var uiEverTag = {
    // Current command being configured
    currentCommand: null,

    /**
     * Initialize the EverTag UI
     */
    initialize: function() {
        // Hide all command config sections initially
        document.getElementById('setled_config').classList.add('w3-hide');
        document.getElementById('readbattery_config').classList.add('w3-hide');
    },

    /**
     * Handle command selection from dropdown
     * @param {string} command - The selected command ('setled' or 'readbattery')
     */
    onCommandSelected: function(command) {
        // Hide all command config sections
        document.getElementById('setled_config').classList.add('w3-hide');
        document.getElementById('readbattery_config').classList.add('w3-hide');

        // Show the selected command's config section
        if (command === 'setled') {
            document.getElementById('setled_config').classList.remove('w3-hide');
        } else if (command === 'readbattery') {
            document.getElementById('readbattery_config').classList.remove('w3-hide');
        }

        this.currentCommand = command;
    },

    /**
     * Send Set LED command to the EverTag
     */
    sendSetLedCommand: function() {
        const ledState = document.getElementById('led_state').value;
        const duration = document.getElementById('led_duration').value;

        // Create the command payload
        const payload = new Uint8Array([
            0x01,  // Command ID for Set LED
            parseInt(ledState),
            parseInt(duration)
        ]);

        // Send the command using the existing message sending mechanism
        this.sendCommand(payload);
    },

    /**
     * Send Read Battery command to the EverTag
     */
    sendReadBatteryCommand: function() {
        // Create the command payload
        const payload = new Uint8Array([
            0x02  // Command ID for Read Battery
        ]);

        // Send the command using the existing message sending mechanism
        this.sendCommand(payload);
    },

    /**
     * Send command to the EverTag
     * @param {Uint8Array} payload - The command payload
     */
    sendCommand: function(payload) {
        // Get the current gateway and sink IDs from the UI
        const gwId = document.getElementById('gw_id_downlink').value;
        const sinkId = document.getElementById('sink_id_downlink').value;
        const destAdd = document.getElementById('dest_add_downlink').value;
        const srcEp = document.getElementById('src_ep_downlink').value;
        const destEp = document.getElementById('dest_ep_downlink').value;
        const qos = document.getElementById('qos_downlink').value;

        // Convert payload to hex string
        const payloadHex = Array.from(payload)
            .map(b => b.toString(16).padStart(2, '0'))
            .join(' ');

        // Create the message object
        const message = {
            gw_id: gwId,
            sink_id: sinkId,
            dest_add: destAdd,
            src_ep: srcEp,
            dest_ep: destEp,
            qos: qos,
            payload: payloadHex
        };

        // Send the message using the existing message sending mechanism
        mainApp.sendMessage(message);

        // Update response text
        document.getElementById('evertag_response_text').value = 
            `Command sent:\nGateway: ${gwId}\nSink: ${sinkId}\nDestination: ${destAdd}\nPayload: ${payloadHex}`;
    },

    /**
     * Handle response from EverTag
     * @param {Object} response - The response message
     */
    handleResponse: function(response) {
        const responseText = document.getElementById('evertag_response_text');
        
        if (response.payload) {
            // Parse the response payload
            const payload = response.payload.split(' ').map(hex => parseInt(hex, 16));
            
            if (payload[0] === 0x02) { // Read Battery response
                const voltage = (payload[1] << 8 | payload[2]) / 1000.0;
                responseText.value = `Battery Voltage: ${voltage.toFixed(3)}V`;
            } else if (payload[0] === 0x01) { // Set LED response
                responseText.value = `LED command response: ${payload[1] === 0 ? 'Success' : 'Failed'}`;
            }
        }
    }
};

// Initialize the EverTag UI when the document is loaded
document.addEventListener('DOMContentLoaded', function() {
    uiEverTag.initialize();
}); 