/**
 * ui_connection_enhanced.js
 *
 * Extends ui_connection.js to add enhanced connection management features.
 * - Adds persistent storage for connection parameters.
 * - May override or extend connection event handlers.
 * - See README.md for extension patterns.
 */

// Initialize the enhanced connection UI when the document is ready
document.addEventListener('DOMContentLoaded', function() {
    initializeEnhancedConnectionUI();
});

/**
 * Initialize the enhanced connection UI
 */
function initializeEnhancedConnectionUI() {
    // Add the "Save credentials" checkbox to the connection form
    addSaveCredentialsCheckbox();
    
    // Load saved credentials if available
    loadSavedCredentials();
    
    // Add event listeners
    addConnectionFormEventListeners();
}

/**
 * Add the "Save credentials" checkbox to the connection form
 */
function addSaveCredentialsCheckbox() {
    // Find the connection form
    const connectionForm = document.forms.mqtt_connection;
    if (!connectionForm) {
        console.error('Connection form not found');
        return;
    }
    
    // Create the checkbox container
    const checkboxContainer = document.createElement('div');
    checkboxContainer.className = 'w3-section';
    
    // Create the checkbox input
    const checkbox = document.createElement('input');
    checkbox.type = 'checkbox';
    checkbox.id = 'save_credentials';
    checkbox.name = 'save_credentials';
    checkbox.className = 'w3-check';
    
    // Create the label
    const label = document.createElement('label');
    label.htmlFor = 'save_credentials';
    label.textContent = ' Save credentials';
    label.style.marginLeft = '8px';
    
    // Append elements
    checkboxContainer.appendChild(checkbox);
    checkboxContainer.appendChild(label);
    
    // Find the reset button container and insert before it
    const resetButton = connectionForm.querySelector('button[type="reset"]');
    if (resetButton && resetButton.parentNode) {
        resetButton.parentNode.parentNode.insertBefore(checkboxContainer, resetButton.parentNode);
    } else {
        // Fallback: append to the form
        connectionForm.appendChild(checkboxContainer);
    }
}

/**
 * Load saved credentials if available
 */
function loadSavedCredentials() {
    // Check if secure storage is available and has saved credentials
    if (window.secureStorage && window.secureStorage.has('mqtt_credentials')) {
        const credentials = window.secureStorage.retrieve('mqtt_credentials');
        if (credentials) {
            // Find the connection form
            const connectionForm = document.forms.mqtt_connection;
            if (!connectionForm) {
                console.error('Connection form not found');
                return;
            }
            
            // Fill the form with saved credentials
            if (credentials.host) connectionForm.mqtt_host.value = credentials.host;
            if (credentials.port) connectionForm.mqtt_port.value = credentials.port;
            if (credentials.ssl !== undefined) connectionForm.mqtt_use_ssl.checked = credentials.ssl;
            if (credentials.username) connectionForm.mqtt_username.value = credentials.username;
            if (credentials.password) connectionForm.mqtt_password.value = credentials.password;
            
            // Check the "Save credentials" checkbox
            const saveCredentialsCheckbox = document.getElementById('save_credentials');
            if (saveCredentialsCheckbox) {
                saveCredentialsCheckbox.checked = true;
            }
        }
    }
}

/**
 * Add event listeners to the connection form
 */
function addConnectionFormEventListeners() {
    // Find the connection form
    const connectionForm = document.forms.mqtt_connection;
    if (!connectionForm) {
        console.error('Connection form not found');
        return;
    }
    
    // Add event listener to the connect button
    const connectButton = document.getElementById('connect_mqtt');
    if (connectButton) {
        connectButton.addEventListener('click', function(event) {
            // Check if credentials should be saved
            const saveCredentialsCheckbox = document.getElementById('save_credentials');
            if (saveCredentialsCheckbox && saveCredentialsCheckbox.checked && window.secureStorage) {
                // Save credentials
                const credentials = {
                    host: connectionForm.mqtt_host.value,
                    port: connectionForm.mqtt_port.value,
                    ssl: connectionForm.mqtt_use_ssl.checked,
                    username: connectionForm.mqtt_username.value,
                    password: connectionForm.mqtt_password.value
                };
                
                window.secureStorage.store('mqtt_credentials', credentials);
            }
        });
    }
    
    // Add event listener to the reset button
    const resetButton = connectionForm.querySelector('button[type="reset"]');
    if (resetButton) {
        resetButton.addEventListener('click', function(event) {
            // Clear saved credentials
            if (window.secureStorage) {
                window.secureStorage.remove('mqtt_credentials');
            }
            
            // Uncheck the "Save credentials" checkbox
            const saveCredentialsCheckbox = document.getElementById('save_credentials');
            if (saveCredentialsCheckbox) {
                saveCredentialsCheckbox.checked = false;
            }
        });
    }
}

// Add a method to check if the enhanced connection UI is available
window.isEnhancedConnectionUIAvailable = function() {
    return true;
}; 