/**
 * secure_storage.js
 * 
 * This file provides secure storage functionality for the Wirepas MQTT Console.
 * It uses the browser's localStorage with basic encryption for storing sensitive data.
 */

// Simple encryption key (in a real application, this would be more secure)
const STORAGE_KEY_PREFIX = 'wirepas_mqtt_console_';
const ENCRYPTION_KEY = 'wirepas_secure_storage';

/**
 * Encrypt a string using a simple XOR cipher
 * @param {string} text - Text to encrypt
 * @param {string} key - Encryption key
 * @returns {string} - Encrypted text
 */
function encryptText(text, key) {
    let result = '';
    for (let i = 0; i < text.length; i++) {
        result += String.fromCharCode(text.charCodeAt(i) ^ key.charCodeAt(i % key.length));
    }
    return btoa(result); // Base64 encode
}

/**
 * Decrypt a string using a simple XOR cipher
 * @param {string} encrypted - Encrypted text (Base64 encoded)
 * @param {string} key - Encryption key
 * @returns {string} - Decrypted text
 */
function decryptText(encrypted, key) {
    try {
        const decoded = atob(encrypted); // Base64 decode
        let result = '';
        for (let i = 0; i < decoded.length; i++) {
            result += String.fromCharCode(decoded.charCodeAt(i) ^ key.charCodeAt(i % key.length));
        }
        return result;
    } catch (e) {
        console.error('Error decrypting data:', e);
        return null;
    }
}

/**
 * Store data securely in localStorage
 * @param {string} key - Storage key
 * @param {any} data - Data to store (will be JSON stringified)
 */
function secureStore(key, data) {
    const fullKey = STORAGE_KEY_PREFIX + key;
    const dataString = JSON.stringify(data);
    const encrypted = encryptText(dataString, ENCRYPTION_KEY);
    localStorage.setItem(fullKey, encrypted);
}

/**
 * Retrieve data from secure storage
 * @param {string} key - Storage key
 * @returns {any} - Retrieved data (JSON parsed) or null if not found
 */
function secureRetrieve(key) {
    const fullKey = STORAGE_KEY_PREFIX + key;
    const encrypted = localStorage.getItem(fullKey);
    
    if (!encrypted) {
        return null;
    }
    
    const decrypted = decryptText(encrypted, ENCRYPTION_KEY);
    if (!decrypted) {
        return null;
    }
    
    try {
        return JSON.parse(decrypted);
    } catch (e) {
        console.error('Error parsing stored data:', e);
        return null;
    }
}

/**
 * Remove data from secure storage
 * @param {string} key - Storage key
 */
function secureRemove(key) {
    const fullKey = STORAGE_KEY_PREFIX + key;
    localStorage.removeItem(fullKey);
}

/**
 * Check if a key exists in secure storage
 * @param {string} key - Storage key
 * @returns {boolean} - True if key exists
 */
function secureHas(key) {
    const fullKey = STORAGE_KEY_PREFIX + key;
    return localStorage.getItem(fullKey) !== null;
}

// Export functions for use in other modules
window.secureStorage = {
    store: secureStore,
    retrieve: secureRetrieve,
    remove: secureRemove,
    has: secureHas
}; 